/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.example.lint.checks;

import com.android.tools.lint.client.api.UElementHandler;
import com.android.tools.lint.detector.api.Category;
import com.android.tools.lint.detector.api.Detector;
import com.android.tools.lint.detector.api.Detector.UastScanner;
import com.android.tools.lint.detector.api.Implementation;
import com.android.tools.lint.detector.api.Issue;
import com.android.tools.lint.detector.api.JavaContext;
import com.android.tools.lint.detector.api.Scope;
import com.android.tools.lint.detector.api.Severity;
import java.util.Collections;
import java.util.List;
import org.jetbrains.uast.UElement;
import org.jetbrains.uast.ULiteralExpression;
import org.jetbrains.uast.UastLiteralUtils;

/**
 * Sample detector showing how to analyze Kotlin/Java code. This example flags all string literals
 * in the code that contain the word "lint".
 */
public class SampleCodeDetectorB extends Detector implements UastScanner {
    /** Issue describing the problem and pointing to the detector implementation */
    public static final Issue ISSUE =
            Issue.create(
                    // ID: used in @SuppressLint warnings etc
                    "ShortUniqueIdB",

                    // Title -- shown in the IDE's preference dialog, as category headers in the
                    // Analysis results window, etc
                    "Lint Mentions B",

                    // Full explanation of the issue; you can use some markdown markup such as
                    // `monospace`, *italic*, and **bold**.
                    "B This check highlights string literals in code which mentions "
                            + "the word `lint`. Blah blah blah.\n"
                            + "\n"
                            + "Another paragraph here.\n",
                    Category.CORRECTNESS,
                    6,
                    Severity.WARNING,
                    new Implementation(SampleCodeDetectorB.class, Scope.JAVA_FILE_SCOPE));

    @Override
    public List<Class<? extends UElement>> getApplicableUastTypes() {
        return Collections.singletonList(ULiteralExpression.class);
    }

    @Override
    public UElementHandler createUastHandler(JavaContext context) {
        // Note: Visiting UAST nodes is a pretty general purpose mechanism;
        // Lint has specialized support to do common things like "visit every class
        // that extends a given super class or implements a given interface", and
        // "visit every call site that calls a method by a given name" etc.
        // Take a careful look at UastScanner and the various existing lint check
        // implementations before doing things the "hard way".
        // Also be aware of context.getJavaEvaluator() which provides a lot of
        // utility functionality.
        return new UElementHandler() {
            @Override
            public void visitLiteralExpression(ULiteralExpression expression) {
                String string = UastLiteralUtils.getValueIfStringLiteral(expression);
                if (string == null) {
                    return;
                }

                if (string.contains("lint") && string.matches(".*\\blint\\b.*")) {
                    context.report(
                            ISSUE,
                            expression,
                            context.getLocation(expression),
                            "B This code mentions `lint`: **Congratulations**");
                }
            }
        };
    }
}
