package com.example.bytecode.plugins;

import com.android.build.gradle.*;
import com.android.build.gradle.api.BaseVariant;
import com.android.build.gradle.api.SourceKind;
import java.io.File;
import java.util.List;
import org.gradle.api.DefaultTask;
import org.gradle.api.DomainObjectSet;
import org.gradle.api.Plugin;
import org.gradle.api.Project;
import org.gradle.api.Task;
import org.gradle.api.file.ConfigurableFileCollection;
import org.gradle.api.file.ConfigurableFileTree;

/**
 * Simple plugin that register a bytecode generating task
 *
 * <p>It loops on all the variants (including the test ones), and for each variant it registers a
 * new task. The bytecode generated by the task depends on the type of the variant (app vs lib vs
 * test)
 */
public class CompilerPlugin implements Plugin<Project> {

    private Project project;
    private File appJar;
    private File postJavacAppJar;
    private File libJar;
    private File postJavacLibJar;
    private File testJar;

    @Override
    public void apply(Project project) {
        this.project = project;

        computeJarLocations();

        // process existing plugins
        project.getPlugins().forEach(this::handlePlugin);
        // process future plugins
        project.getPlugins().whenPluginAdded(this::handlePlugin);
    }

    private void handlePlugin(Plugin plugin) {
        final Class<? extends Plugin> pluginClass = plugin.getClass();

        TestedExtension testedExtension = null;

        if (pluginClass.equals(AppPlugin.class)) {
            AppExtension extension = (AppExtension) project.getExtensions().getByName("android");
            processVariants(extension.getApplicationVariants(), appJar, postJavacAppJar);

            testedExtension = extension;
        } else if (pluginClass.equals(LibraryPlugin.class)) {
            LibraryExtension extension =
                    (LibraryExtension) project.getExtensions().getByName("android");
            processVariants(extension.getLibraryVariants(), libJar, postJavacLibJar);

            testedExtension = extension;
        } else if (pluginClass.equals(TestPlugin.class)) {
            TestExtension extension = (TestExtension) project.getExtensions().getByName("android");
            processVariants(extension.getApplicationVariants(), testJar, null);
            // no test here.
        }

        if (testedExtension != null) {
            processVariants(testedExtension.getTestVariants(), testJar, null);
            processVariants(testedExtension.getUnitTestVariants(), testJar, null);
        }
    }

    private void computeJarLocations() {
        File rootDir = project.getRootProject().getProjectDir();
        appJar = new File(rootDir, "app.jar");
        postJavacAppJar = new File(rootDir, "postjavacapp.jar");
        libJar = new File(rootDir, "lib.jar");
        postJavacLibJar = new File(rootDir, "postjavaclib.jar");
        testJar = new File(rootDir, "test.jar");
    }

    private <T extends BaseVariant> void processVariants(
            DomainObjectSet<T> variants, File sourceJar, File postJavacJar) {

        File folder = project.file("src/custom/java");
        Task findTask = project.getTasks().findByName("generateCustomTaskForTesting");
        if (findTask == null) {
            findTask = project.getTasks().create("generateCustomTaskForTesting", DefaultTask.class);
        }

        // need to be final because lambda below
        Task sourceTask = findTask;

        variants.all(
                variant -> {
                    if (postJavacJar != null) {
                        variant.registerJavaGeneratingTask(sourceTask, folder);
                    }

                    // get the source folders.
                    List<ConfigurableFileTree> sourceFolders =
                            variant.getSourceFolders(SourceKind.JAVA);

                    // output the source folder gotten by the API
                    for (ConfigurableFileTree fileTree : sourceFolders) {
                        System.out.println(
                                "SourceFoldersApi("
                                        + project.getPath()
                                        + ":"
                                        + variant.getName()
                                        + "): "
                                        + fileTree.getDir());
                    }

                    // figure out the output.
                    File outputDir =
                            project.file(
                                    project.getBuildDir()
                                            + "/generated/preJavacbytecode/"
                                            + variant.getDirName());

                    // create the file collection that contains the result. We'll add the task
                    // dependency later when we have it
                    ConfigurableFileCollection fc = project.files(outputDir);

                    // and register it with the variant, getting the key in return that will
                    // be used for the classpath
                    Object key = variant.registerPreJavacGeneratedBytecode(fc);

                    // create the task, querying the classpath with the provided key.
                    BytecodeGeneratingTask t =
                            project.getTasks()
                                    .create(
                                            "generateBytecodeFor" + variant.getName(),
                                            BytecodeGeneratingTask.class,
                                            task -> {
                                                task.setSourceFolders(sourceFolders);
                                                task.setSourceJar(sourceJar);
                                                task.setOutputDir(outputDir);
                                                task.setClasspath(
                                                        variant.getCompileClasspathArtifacts(key));
                                            });

                    // add the task dependency to the file collection so that consumers can have
                    // the proper dependency.
                    fc.builtBy(t);

                    // make the task run after the variant's prebuild task
                    t.dependsOn(variant.getPreBuild());

                    // also create a post javac bytecode generating task if needed
                    if (postJavacJar != null) {
                        File outputDir2 =
                                project.file(
                                        project.getBuildDir()
                                                + "/generated/postJavacBytecode/"
                                                + variant.getDirName());

                        BytecodeGeneratingTask t2 =
                                project.getTasks()
                                        .create(
                                                "generateBytecode2For" + variant.getName(),
                                                BytecodeGeneratingTask.class,
                                                task -> {
                                                    task.setSourceFolders(sourceFolders);
                                                    task.setSourceJar(postJavacJar);
                                                    task.setOutputDir(outputDir2);
                                                });

                        ConfigurableFileCollection fc2 = project.files(outputDir2).builtBy(t2);
                        variant.registerPostJavacGeneratedBytecode(fc2);
                    }
                });
    }
}
