/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.integration.common.fixture;

import com.android.SdkConstants;
import com.android.annotations.NonNull;
import com.android.annotations.Nullable;
import com.android.io.FolderWrapper;
import com.android.io.IAbstractFile;
import com.android.io.IAbstractFolder;
import com.android.io.StreamException;
import com.android.sdklib.internal.project.ProjectProperties;
import com.android.utils.SdkUtils;
import com.google.common.io.Closeables;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A modifiable and savable copy of a {@link ProjectProperties}.
 *
 * <p>This copy gives access to modification method such as {@link #setProperty(String, String)}
 *
 * <p>To get access to an instance, use {@link #create(String, PropertyType)}.
 */
class ProjectPropertiesWorkingCopy extends ProjectProperties {

    private static final Map<String, String> COMMENT_MAP = new HashMap<>();

    static {
//               1-------10--------20--------30--------40--------50--------60--------70--------80
        COMMENT_MAP.put(PROPERTY_TARGET,
                "# Project target.\n");
        COMMENT_MAP.put(PROPERTY_SPLIT_BY_DENSITY,
                "# Indicates whether an apk should be generated for each density.\n");
        COMMENT_MAP.put(PROPERTY_SDK,
                "# location of the SDK. This is only used by Ant\n" +
                "# For customization when using a Version Control System, please read the\n" +
                "# header note.\n");
        COMMENT_MAP.put(PROPERTY_PACKAGE,
                "# Package of the application being exported\n");
        COMMENT_MAP.put(PROPERTY_VERSIONCODE,
                "# Major version code\n");
        COMMENT_MAP.put(PROPERTY_PROJECTS,
                "# List of the Android projects being used for the export.\n" +
                "# The list is made of paths that are relative to this project,\n" +
                "# using forward-slash (/) as separator, and are separated by colons (:).\n");
    }

    // LEGACY - Kept so that we can actually remove it from local.properties.
    private static final String PROPERTY_SDK_LEGACY = "sdk-location";

    private static final String LOCAL_HEADER =
            //
            // 1-------10--------20--------30--------40--------50--------60--------70--------80
            "# This file is automatically generated by Android Tools.\n"
                    + "# Do not modify this file -- YOUR CHANGES WILL BE ERASED!\n"
                    + "#\n"
                    + "# This file must *NOT* be checked into Version Control Systems,\n"
                    + "# as it contains information specific to your local configuration.\n"
                    + "\n";

    public enum PropertyType {
        LOCAL(
                SdkConstants.FN_LOCAL_PROPERTIES,
                LOCAL_HEADER,
                new String[] {PROPERTY_SDK},
                new String[] {PROPERTY_SDK_LEGACY});

        private final String mFilename;
        private final String mHeader;
        private final Set<String> mKnownProps;
        private final Set<String> mRemovedProps;

        PropertyType(
                String filename,
                String header,
                @Nullable String[] validProps,
                @Nullable String[] removedProps) {
            mFilename = filename;
            mHeader = header;
            HashSet<String> s = new HashSet<>();
            if (validProps != null) {
                s.addAll(Arrays.asList(validProps));
            }
            mKnownProps = Collections.unmodifiableSet(s);

            s = new HashSet<>();
            if (removedProps != null) {
                s.addAll(Arrays.asList(removedProps));
            }
            mRemovedProps = Collections.unmodifiableSet(s);
        }

        public String getFilename() {
            return mFilename;
        }

        public String getHeader() {
            return mHeader;
        }

        /** Returns whether a given property is known for the property type. */
        public boolean isKnownProperty(String name) {
            for (String propRegex : mKnownProps) {
                if (propRegex.equals(name) || Pattern.matches(propRegex, name)) {
                    return true;
                }
            }

            return false;
        }

        /** Returns whether a given property should be removed for the property type. */
        public boolean isRemovedProperty(String name) {
            for (String propRegex : mRemovedProps) {
                if (propRegex.equals(name) || Pattern.matches(propRegex, name)) {
                    return true;
                }
            }

            return false;
        }
    }

    protected final IAbstractFolder mProjectFolder;
    protected final Map<String, String> mProperties;
    protected final PropertyType mType;

    /**
     * Creates a new project properties object, with no properties.
     *
     * <p>The file is not created until {@link ProjectPropertiesWorkingCopy#save()} is called.
     *
     * @param projectFolderOsPath the project folder.
     * @param type the type of property file to create
     */
    public static ProjectPropertiesWorkingCopy create(
            @NonNull String projectFolderOsPath, @NonNull PropertyType type) {
        // create and return a ProjectProperties with an empty map.
        IAbstractFolder folder = new FolderWrapper(projectFolderOsPath);
        return new ProjectPropertiesWorkingCopy(folder, new HashMap<>(), type);
    }

    /** Returns the location of this property file. */
    public IAbstractFile getFile() {
        return mProjectFolder.getFile(mType.mFilename);
    }

    /**
     * Sets a new properties. If a property with the same name already exists, it is replaced.
     *
     * @param name the name of the property.
     * @param value the value of the property.
     */
    public synchronized void setProperty(String name, String value) {
        mProperties.put(name, value);
    }

    /**
     * Saves the property file, using UTF-8 encoding.
     */
    public synchronized void save() throws IOException, StreamException {
        IAbstractFile toSave = mProjectFolder.getFile(mType.getFilename());

        // write the whole file in a byte array before dumping it in the file. This
        // This is so that if the file already existing
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OutputStreamWriter writer = new OutputStreamWriter(baos, SdkConstants.INI_CHARSET);

        if (toSave.exists()) {
            InputStream contentStream = toSave.getContents();
            InputStreamReader isr = null;
            BufferedReader reader = null;

            try {
                contentStream = toSave.getContents();
                isr = new InputStreamReader(contentStream, SdkConstants.INI_CHARSET);
                reader = new BufferedReader(isr);

                // since we're reading the existing file and replacing values with new ones, or
                // skipping
                // removed values, we need to record what properties have been visited, so that
                // we can figure later what new properties need to be added at the end of the file.
                Set<String> visitedProps = new HashSet<>();

                String line;
                while ((line = reader.readLine()) != null) {
                    // check if this is a line containing a property.
                    if (!line.isEmpty() && line.charAt(0) != '#') {

                        Matcher m = PATTERN_PROP.matcher(line);
                        if (m.matches()) {
                            String key = m.group(1);
                            String value = m.group(2);

                            // record the prop
                            visitedProps.add(key);

                            // check if this property must be removed.
                            if (mType.isRemovedProperty(key)) {
                                value = null;
                            } else if (mProperties.containsKey(key)) { // if the property still exists.
                                // put the new value.
                                value = mProperties.get(key);
                            } else {
                                // property doesn't exist. Check if it's a known property.
                                // if it's a known one, we'll remove it, otherwise, leave it untouched.
                                if (mType.isKnownProperty(key)) {
                                    value = null;
                                }
                            }

                            // if the value is still valid, write it down.
                            if (value != null) {
                                writeValue(writer, key, value, false /*addComment*/);
                            }
                        } else  {
                            // the line was wrong, let's just ignore it so that it's removed from the
                            // file.
                        }
                    } else {
                        // non-property line: just write the line in the output as-is.
                        writer.append(line).append('\n');
                    }
                }

                // now add the new properties.
                for (Entry<String, String> entry : mProperties.entrySet()) {
                    if (!visitedProps.contains(entry.getKey())) {
                        String value = entry.getValue();
                        if (value != null) {
                            writeValue(writer, entry.getKey(), value, true /*addComment*/);
                        }
                    }
                }
            } finally {
                try {
                    Closeables.close(reader, true /* swallowIOException */);
                } catch (IOException e) {
                    // cannot happen
                }
                try {
                    Closeables.close(isr, true /* swallowIOException */);
                } catch (IOException e) {
                    // cannot happen
                }
                try {
                    Closeables.close(contentStream, true /* swallowIOException */);
                } catch (IOException e) {
                    // cannot happen
                }
            }

        } else {
            // new file, just write it all

            // write the header (can be null, for example for PropertyType.LEGACY_BUILD)
            if (mType.getHeader() != null) {
                writer.write(mType.getHeader());
            }

            // write the properties.
            for (Entry<String, String> entry : mProperties.entrySet()) {
                String value = entry.getValue();
                if (value != null) {
                    writeValue(writer, entry.getKey(), value, true /*addComment*/);
                }
            }
        }

        writer.flush();

        // now put the content in the file.
        OutputStream filestream = toSave.getOutputStream();
        filestream.write(baos.toByteArray());
        filestream.flush();
        filestream.close();
    }

    private static void writeValue(
            OutputStreamWriter writer, String key, String value, boolean addComment)
            throws IOException {
        if (addComment) {
            String comment = COMMENT_MAP.get(key);
            if (comment != null) {
                writer.write(comment);
            }
        }

        writer.write(String.format("%s=%s\n", key, SdkUtils.escapePropertyValue(value)));
    }

    /**
     * Private constructor.
     *
     * <p>Use {@link #create(String, PropertyType)} to instantiate.
     */
    private ProjectPropertiesWorkingCopy(
            IAbstractFolder projectFolder, Map<String, String> map, PropertyType type) {
        mProjectFolder = projectFolder;
        mProperties = map;
        mType = type;
    }
}
