/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.integration.common.fixture;

import com.android.SdkConstants;
import com.android.annotations.NonNull;
import com.android.annotations.Nullable;
import com.android.build.gradle.integration.common.fixture.gradle_project.ProjectLocation;
import com.android.build.gradle.integration.common.utils.JacocoAgent;
import com.android.build.gradle.options.BooleanOption;
import com.android.build.gradle.options.IntegerOption;
import com.android.build.gradle.options.Option;
import com.android.build.gradle.options.OptionalBooleanOption;
import com.android.build.gradle.options.StringOption;
import com.android.prefs.AbstractAndroidLocations;
import com.android.testutils.TestUtils;
import com.android.utils.FileUtils;
import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import com.google.common.util.concurrent.SettableFuture;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import kotlin.io.FilesKt;
import org.apache.commons.io.output.TeeOutputStream;
import org.gradle.tooling.CancellationTokenSource;
import org.gradle.tooling.ConfigurableLauncher;
import org.gradle.tooling.GradleConnectionException;
import org.gradle.tooling.GradleConnector;
import org.gradle.tooling.LongRunningOperation;
import org.gradle.tooling.ProjectConnection;
import org.gradle.tooling.ResultHandler;
import org.gradle.tooling.events.ProgressEvent;
import org.gradle.tooling.events.ProgressListener;

/**
 * Common flags shared by {@link ModelBuilderV2} and {@link GradleTaskExecutor}.
 *
 * @param <T> The concrete implementing class.
 */
@SuppressWarnings("unchecked") // Returning this as <T> in most methods.
public abstract class BaseGradleExecutor<T extends BaseGradleExecutor> {

    // An internal timeout for executing Gradle. This aims to be less than the overall test timeout
    // to give more instructive error messages
    private static final long TIMEOUT_SECONDS;

    private static Path jvmLogDir;

    private static final Path jvmErrorLog;

    static {
        String timeoutOverride = System.getenv("TEST_TIMEOUT");
        if (timeoutOverride != null) {
            // Allow for longer build times within a test, while still trying to avoid having the
            // overal test timeout be hit. If TEST_TIMEOUT is set, potentially increase the timeout
            // to 1 minute less than the overall test timeout, if that's more than the default 10
            // minute timeout.
            TIMEOUT_SECONDS = Math.max(600, Integer.parseInt(timeoutOverride) - 60);
        } else {
            TIMEOUT_SECONDS = 600;
        }
        try {
            jvmLogDir = Files.createTempDirectory("GRADLE_JVM_LOGS");
            jvmErrorLog = jvmLogDir.resolve("java_error.log");
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static final boolean VERBOSE =
            !Strings.isNullOrEmpty(System.getenv().get("CUSTOM_TEST_VERBOSE"));
    static final boolean CAPTURE_JVM_LOGS = false;

    @NonNull
    final ProjectConnection projectConnection;
    @Nullable protected final GradleTestRule project;
    @NonNull public final ProjectLocation projectLocation;
    @NonNull final Consumer<GradleBuildResult> lastBuildResultConsumer;
    @NonNull private final List<String> arguments = Lists.newArrayList();
    @NonNull private final ProjectOptionsBuilder options = new ProjectOptionsBuilder();
    @NonNull private final GradleTestProjectBuilder.MemoryRequirement memoryRequirement;
    @NonNull private LoggingLevel loggingLevel = LoggingLevel.INFO;
    private boolean offline = true;
    private boolean localPrefsRoot = false;
    private boolean perTestPrefsRoot = false;
    private boolean failOnWarning = true;

    private boolean crashOnOutOfMemory = false;
    private ConfigurationCaching configurationCaching;

    BaseGradleExecutor(
            @Nullable GradleTestRule project,
            @NonNull ProjectLocation projectLocation,
            @NonNull ProjectConnection projectConnection,
            @NonNull Consumer<GradleBuildResult> lastBuildResultConsumer,
            @Nullable Path profileDirectory,
            @NonNull GradleTestProjectBuilder.MemoryRequirement memoryRequirement,
            @NonNull ConfigurationCaching configurationCaching) {
        this.project = project;
        this.projectLocation = projectLocation;
        this.lastBuildResultConsumer = lastBuildResultConsumer;
        this.projectConnection = projectConnection;
        this.memoryRequirement = memoryRequirement;
        this.configurationCaching = configurationCaching;

        if (profileDirectory != null) {
            with(StringOption.PROFILE_OUTPUT_DIR, profileDirectory.toString());
        }
    }

    public final T with(@NonNull BooleanOption option, boolean value) {
        options.booleans.put(option, value);
        return (T) this;
    }

    public final T with(@NonNull OptionalBooleanOption option, boolean value) {
        options.optionalBooleans.put(option, value);
        return (T) this;
    }

    public final T with(@NonNull IntegerOption option, int value) {
        options.integers.put(option, value);
        return (T) this;
    }

    public final T with(@NonNull StringOption option, @NonNull String value) {
        options.strings.put(option, value);
        return (T) this;
    }

    public final T suppressOptionWarning(@NonNull Option option) {
        options.suppressWarnings.add(option);
        return (T) this;
    }

    @Deprecated
    @NonNull
    public T withProperty(@NonNull String propertyName, @NonNull String value) {
        withArgument("-P" + propertyName + "=" + value);
        return (T) this;
    }

    /** Add additional build arguments. */
    public final T withArguments(@NonNull List<String> arguments) {
        for (String argument : arguments) {
            withArgument(argument);
        }
        return (T) this;
    }

    /** Add an additional build argument. */
    public final T withArgument(String argument) {
        if (argument.startsWith("-Pandroid")
                && !argument.contains("testInstrumentationRunnerArguments")) {
            throw new IllegalArgumentException("Use with(Option, Value) instead.");
        }
        arguments.add(argument);
        return (T) this;
    }

    public T withEnableInfoLogging(boolean enableInfoLogging) {
        return withLoggingLevel(enableInfoLogging ? LoggingLevel.INFO : LoggingLevel.LIFECYCLE);
    }

    public T withLoggingLevel(@NonNull LoggingLevel loggingLevel) {
        this.loggingLevel = loggingLevel;
        return (T) this;
    }

    /** Sets to run Gradle with the normal preference root (~/.android) */
    public final T withLocalPrefsRoot() {
        localPrefsRoot = true;
        return (T) this;
    }

    /**
     * Sets whether to run Gradle with a per-test preference root.
     *
     * <p>The preference root outside of test is normally ~/.android.
     *
     * <p>If set to false, the folder is located in the build output, common to all tests.
     *
     * <p>If set to true, the test will use its own isolated folder.
     */
    public final T withPerTestPrefsRoot(boolean perTestPrefsRoot) {
        this.perTestPrefsRoot = perTestPrefsRoot;
        return (T) this;
    }

    public final T withoutOfflineFlag() {
        this.offline = false;
        return (T) this;
    }

    public final T withSdkAutoDownload() {
        return with(BooleanOption.ENABLE_SDK_DOWNLOAD, true);
    }

    public final T withFailOnWarning(boolean failOnWarning) {
        this.failOnWarning = failOnWarning;
        return (T) this;
    }

    public final T withConfigurationCaching(ConfigurationCaching configurationCaching) {
        this.configurationCaching = configurationCaching;
        return (T) this;
    }

    /** Forces JVM exit in the event of an OutOfMemoryError, without collecting a heap dump. */
    public final T crashOnOutOfMemory() {
        this.crashOnOutOfMemory = true;
        return (T) this;
    }

    protected final List<String> getArguments() throws IOException {
        List<String> arguments = new ArrayList<>();
        arguments.addAll(this.arguments);
        arguments.addAll(options.getArguments());

        if (loggingLevel.getArgument() != null) {
            arguments.add(loggingLevel.getArgument());
        }

        arguments.add("-Dfile.encoding=" + System.getProperty("file.encoding"));
        arguments.add("-Dsun.jnu.encoding=" + System.getProperty("sun.jnu.encoding"));

        if (offline) {
            arguments.add("--offline");
        }
        if (failOnWarning) {
            arguments.add("--warning-mode=fail");
        }

        switch (configurationCaching) {
            case ON:
                arguments.add("--configuration-cache");
                arguments.add("--configuration-cache-problems=fail");
                break;
            case PROJECT_ISOLATION:
                arguments.add("-Dorg.gradle.unsafe.isolated-projects=true");
                arguments.add("--configuration-cache-problems=warn");
                break;
            case OFF:
                arguments.add("--no-configuration-cache");
                break;
        }

        if (!localPrefsRoot) {
            File preferencesRootDir;
            if (perTestPrefsRoot) {
                preferencesRootDir =
                        new File(
                                projectLocation.getProjectDir().getParentFile(),
                                "android_prefs_root");
            } else {
                preferencesRootDir =
                        new File(
                                projectLocation.getTestLocation().getBuildDir(),
                                "android_prefs_root");
            }

            FileUtils.mkdirs(preferencesRootDir);

            this.preferencesRootDir = preferencesRootDir;

            arguments.add(
                    String.format(
                            "-D%s=%s",
                            AbstractAndroidLocations.ANDROID_PREFS_ROOT,
                            preferencesRootDir.getAbsolutePath()));
        }

        return arguments;
    }

    /*
     * A good-enough heuristic to check if the Kotlin plugin is applied.
     * This is needed because of b/169842093.
     */
    private boolean ifAppliesKotlinPlugin(GradleTestProject testProject) {
        GradleTestProject rootProject = testProject.getRootProject();

        for (File buildFile :
                FileUtils.find(rootProject.getProjectDir(), Pattern.compile("build\\.gradle"))) {
            if (FilesKt.readLines(buildFile, Charsets.UTF_8).stream()
                    .anyMatch(
                            s ->
                                    s.contains("apply plugin: 'kotlin'")
                                            || s.contains("apply plugin: 'kotlin-android'"))) {
                return true;
            }
        }

        return false;
    }

    /** Location of the Android Preferences folder (normally in ~/.android) */
    @Nullable private File preferencesRootDir = null;

    @NonNull
    public File getPreferencesRootDir() {
        if (preferencesRootDir == null) {
            throw new RuntimeException(
                    "cannot call getPreferencesRootDir before it is initialized");
        }

        return preferencesRootDir;
    }

    protected final Set<String> getOptionPropertyNames() {
        return options.getOptions()
                .map(Option::getPropertyName)
                .collect(ImmutableSet.toImmutableSet());
    }

    protected final void setJvmArguments(@NonNull LongRunningOperation launcher) {

        List<String> jvmArguments = new ArrayList<>(this.memoryRequirement.getJvmArgs());

        if (crashOnOutOfMemory) {
            jvmArguments.add("-XX:+CrashOnOutOfMemoryError");
        } else {
            jvmArguments.add("-XX:+HeapDumpOnOutOfMemoryError");
            jvmArguments.add("-XX:HeapDumpPath=" + jvmLogDir.resolve("heapdump.hprof"));
        }

        String debugIntegrationTest = System.getenv("DEBUG_INNER_TEST");
        if (!Strings.isNullOrEmpty(debugIntegrationTest)) {
            String serverArg = debugIntegrationTest.equalsIgnoreCase("socket-listen") ? "n" : "y";
            jvmArguments.add(
                    String.format(
                            "-agentlib:jdwp=transport=dt_socket,server=%s,suspend=y,address=5006",
                            serverArg));
        }

        if (JacocoAgent.isJacocoEnabled()) {
            jvmArguments.add(
                    JacocoAgent.getJvmArg(projectLocation.getTestLocation().getBuildDir()));
        }

        jvmArguments.add("-XX:ErrorFile=" + jvmErrorLog);
        if (CAPTURE_JVM_LOGS) {
            jvmArguments.add("-XX:+UnlockDiagnosticVMOptions");
            jvmArguments.add("-XX:+LogVMOutput");
            jvmArguments.add("-XX:LogFile=" + jvmLogDir.resolve("java_log.log").toString());
        }

        launcher.setJvmArguments(Iterables.toArray(jvmArguments, String.class));
    }

    protected static void setStandardOut(
            @NonNull LongRunningOperation launcher, @NonNull OutputStream stdout) {
        if (VERBOSE) {
            launcher.setStandardOutput(new TeeOutputStream(stdout, System.out));
        } else {
            launcher.setStandardOutput(stdout);
        }
    }

    protected static void setStandardError(
            @NonNull LongRunningOperation launcher, @NonNull OutputStream stderr) {
        if (VERBOSE) {
            launcher.setStandardError(new TeeOutputStream(stderr, System.err));
        } else {
            launcher.setStandardError(stderr);
        }
    }

    @NonNull
    public File getJvmErrorLog() {
        return jvmErrorLog.toFile();
    }

    private void printJvmLogs() throws IOException {

        List<Path> files;
        try (Stream<Path> walk = Files.walk(jvmLogDir)) {
            files = walk.filter(Files::isRegularFile).collect(Collectors.toList());
        }
        if (files.isEmpty()) {
            return;
        }

        Path projectDirectory = projectLocation.getProjectDir().toPath();

        Path outputs;
        if (TestUtils.runningFromBazel()) {

            // Put in test undeclared output directory.
            outputs =
                    TestUtils.getTestOutputDir()
                            .resolve(projectDirectory.getParent().getParent().getFileName())
                            .resolve(projectDirectory.getParent().getFileName())
                            .resolve(projectDirectory.getFileName());
        } else {
            outputs = projectDirectory.resolve("jvm_logs_outputs");
        }
        Files.createDirectories(outputs);

        System.err.println("----------- JVM Log start -----------");
        System.err.println("----- JVM log files being put in " + outputs.toString() + " ----");
        for (Path path : files) {
            System.err.print("---- Copying Log file: ");
            System.err.println(path.getFileName());
            Files.move(path, outputs.resolve(path.getFileName()));
        }
        System.err.println("------------ JVM Log end ------------");
    }

    protected void maybePrintJvmLogs(@NonNull GradleConnectionException failure)
            throws IOException {
        String stacktrace = Throwables.getStackTraceAsString(failure);
        if (stacktrace.contains("org.gradle.launcher.daemon.client.DaemonDisappearedException")
                || stacktrace.contains("java.lang.OutOfMemoryError")) {
                    printJvmLogs();
        }
    }

    protected static class CollectingProgressListener implements ProgressListener {
        final ConcurrentLinkedQueue<ProgressEvent> events;

        protected CollectingProgressListener() {
            events = new ConcurrentLinkedQueue<>();
        }

        @Override
        public void statusChanged(ProgressEvent progressEvent) {
            events.add(progressEvent);
        }

        ImmutableList<ProgressEvent> getEvents() {
            return ImmutableList.copyOf(events);
        }
    }

    protected interface RunAction<LauncherT, ResultT> {
        void run(@NonNull LauncherT launcher, @NonNull ResultHandler<ResultT> resultHandler);
    }

    public enum ConfigurationCaching {
        ON,
        PROJECT_ISOLATION,

        /**
         * Disables configuration cache (i.e., pass `--no-configuration-cache` to the build's
         * arguments).
         *
         * <p>Note: Using this option is not recommended. Only use it if absolutely required.
         */
        @Deprecated
        OFF
    }

    @NonNull
    protected static <LauncherT extends ConfigurableLauncher<LauncherT>, ResultT> ResultT runBuild(
            @NonNull LauncherT launcher, @NonNull RunAction<LauncherT, ResultT> runAction) {
        CancellationTokenSource cancellationTokenSource =
                GradleConnector.newCancellationTokenSource();
        launcher.withCancellationToken(cancellationTokenSource.token());
        SettableFuture<ResultT> future = SettableFuture.create();
        runAction.run(
                launcher,
                new ResultHandler<ResultT>() {
                    @Override
                    public void onComplete(ResultT result) {
                        future.set(result);
                    }

                    @Override
                    public void onFailure(GradleConnectionException e) {
                        future.setException(e);
                    }
                });
        try {
            return future.get(TIMEOUT_SECONDS, TimeUnit.SECONDS);
        } catch (ExecutionException e) {
            throw (GradleConnectionException) e.getCause();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        } catch (TimeoutException e) {
            try {
                printThreadDumps();
            } catch (Throwable t) {
                e.addSuppressed(t);
            }
            cancellationTokenSource.cancel();
            // TODO(b/78568459) Gather more debugging info from Gradle daemon.
            throw new RuntimeException(e);
        }
    }

    private static void printThreadDumps() throws IOException, InterruptedException {
        if (SdkConstants.currentPlatform() != SdkConstants.PLATFORM_LINUX
                && SdkConstants.currentPlatform() != SdkConstants.PLATFORM_DARWIN) {
            // handle only Linux&Darwin for now
            return;
        }
        String javaHome = System.getProperty("java.home");
        String processes = runProcess(javaHome + "/bin/jps");

        String[] lines = processes.split(System.lineSeparator());
        for (String line : lines) {
            String pid = line.split(" ")[0];
            String threadDump = runProcess(javaHome + "/bin/jstack", "-l", pid);

            System.out.println("Fetching thread dump for: " + line);
            System.out.println("Thread dump is:");
            System.out.println(threadDump);
        }
    }

    private static String runProcess(String... commands) throws InterruptedException, IOException {
        ProcessBuilder processBuilder = new ProcessBuilder().command(commands);
        Process process = processBuilder.start();
        process.waitFor(5, TimeUnit.SECONDS);

        byte[] bytes = ByteStreams.toByteArray(process.getInputStream());
        return new String(bytes, Charsets.UTF_8);
    }
}
