/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.utils;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

public abstract class JarOutputCompiler {
    private final String name;
    protected final PrintStream err;

    protected JarOutputCompiler(String name, PrintStream err) {
        this.name = name;
        this.err = err;
    }

    private void usage(String message) {
        err.println("Error: " + message);
        err.println("Usage: " + name + " [-cp class_path|@<file>] -o jar_file <forwardedArgs>");
    }

    protected int run(List<String> args) throws IOException {
        Options options = parseOptions(args.iterator());
        if (options.out == null) {
            usage("Output file not specified.");
            return 1;
        }
        File tmp = new File(options.out.getAbsolutePath() + ".dir");
        if (!compile(options.forwardedArgs, options.classPath, tmp)) {
            return 1;
        }
        new Zipper().directoryToZip(tmp, options.out);
        return 0;
    }

    private Options parseOptions(Iterator<String> it) throws IOException {
        Options options = new Options();
        while (it.hasNext()) {
            String arg = it.next();
            if (arg.equals("-o") && it.hasNext()) {
                options.out = new File(it.next());
            } else if (arg.equals("-cp") && it.hasNext()) {
                String cpArg = it.next();
                if (cpArg.startsWith("@")) {
                    options.classPath = Files.readAllLines(Paths.get(cpArg.substring(1)))
                            .stream()
                            .reduce("", (String a, String b) -> a + b);
                } else {
                    options.classPath = cpArg;
                }
            } else {
                if (arg.startsWith("@")) {
                    options.forwardedArgs.addAll(Files.readAllLines(Paths.get(arg.substring(1))));
                } else {
                    options.forwardedArgs.add(arg);
                }
            }
        }
        return options;
    }

    private static class Options {
        public File out;
        public List<String> forwardedArgs = new LinkedList<>();
        public String classPath = "";
    }

    protected abstract boolean compile(List<String> forwardedArgs, String classPath, File outDir)
            throws IOException;
}
