/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.form;

import com.android.tools.utils.BazelMultiplexWorker;
import com.android.tools.utils.JarOutputCompiler;
import com.android.tools.utils.Unzipper;
import com.intellij.compiler.instrumentation.InstrumentationClassFinder;
import com.intellij.compiler.instrumentation.InstrumenterClassWriter;
import com.intellij.uiDesigner.compiler.AlienFormFileException;
import com.intellij.uiDesigner.compiler.AsmCodeGenerator;
import com.intellij.uiDesigner.compiler.FormErrorInfo;
import com.intellij.uiDesigner.compiler.NestedFormLoader;
import com.intellij.uiDesigner.compiler.Utils;
import com.intellij.uiDesigner.lw.CompiledClassPropertiesProvider;
import com.intellij.uiDesigner.lw.LwRootContainer;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jetbrains.org.objectweb.asm.ClassWriter;

public class FormCompiler extends JarOutputCompiler implements NestedFormLoader {
    private File mOutDir;
    private InstrumentationClassFinder mFinder;
    private final HashMap<String, LwRootContainer> mCache = new HashMap<>();
    private final List<File> mForms = new ArrayList<>();
    private final Map<String, File> mOtherForms = new HashMap<>();

    public FormCompiler(PrintStream err) {
        super("formc", err);
    }

    public static void main(String[] args) throws Exception {
        // Radar #5755208: Command line Java applications need a way to launch without a Dock icon.
        System.setProperty("apple.awt.UIElement", "true");
        BazelMultiplexWorker.run(args, (compilerArgs, out) -> new FormCompiler(out).run(compilerArgs));
    }

    @Override
    protected boolean compile(List<String> forwardedArgs, String classPath, File outDir) throws IOException {
        // Files can contain .jar and .form which will be added together into one output jar.
        mOutDir = outDir;
        final ArrayList<URL> urls = new ArrayList<>();

        addUrlsTo(System.getProperty("java.class.path"), urls);
        addUrlsTo(classPath.replaceAll(":", File.pathSeparator), urls);

        Pattern formPattern = Pattern.compile("(.*)=(.*\\.form)");
        for (String file : forwardedArgs) {
            if (file.endsWith(".jar")) {
                addUrlsTo(file, urls);
                Unzipper unzipper = new Unzipper();
                unzipper.unzip(new File(file), outDir);
            } else if (file.endsWith(".form")) {
                Matcher matcher = formPattern.matcher(file);
                if (matcher.matches()) {
                    mOtherForms.put(matcher.group(1).toLowerCase(), new File(matcher.group(2)));
                } else {
                    mForms.add(new File(file));
                }
            }
        }

        mFinder = new InstrumentationClassFinder(urls.toArray(new URL[0]));
        try {
            return instrumentForms(mFinder, mForms);
        } finally {
            mFinder.releaseResources();
        }
    }

    private void addUrlsTo(String classPath, ArrayList<URL> urls) throws MalformedURLException {
        for (StringTokenizer token = new StringTokenizer(classPath, File.pathSeparator); token.hasMoreTokens(); ) {
            urls.add(new File(token.nextToken()).toURI().toURL());
        }
    }

    private boolean instrumentForms(final InstrumentationClassFinder finder, List<File> forms) {
        if (forms.isEmpty()) {
            throw new IllegalArgumentException("No forms to instrument found");
        }

        final HashMap<String, File> class2form = new HashMap<>();
        for (File form : forms) {
            final LwRootContainer rootContainer;
            try {
                rootContainer = Utils.getRootContainer(form.toURI().toURL(),
                        new CompiledClassPropertiesProvider(finder.getLoader()));
            } catch (AlienFormFileException e) {
                // Ignore non-IDEA forms.
                continue;
            } catch (Exception e) {
                throw new RuntimeException("Cannot process form file " + form.getAbsolutePath(), e);
            }

            final String classToBind = rootContainer.getClassToBind();
            if (classToBind == null) {
                continue;
            }

            File classFile = getClassFile(classToBind);
            if (classFile == null) {
                throw new RuntimeException(String.format("%s: Class to bind does not exist: %s",
                        form.getAbsolutePath(), classToBind));
            }

            final File alreadyProcessedForm = class2form.get(classToBind);
            if (alreadyProcessedForm != null) {
                err.println(String.format(
                        "%s: The form is bound to the class %s.\n" +
                        "Another form %s is also bound to this class.",
                        form.getAbsolutePath(),
                        classToBind,
                        alreadyProcessedForm.getAbsolutePath()));
                continue;
            }
            class2form.put(classToBind, form);

            InstrumenterClassWriter classWriter =
                    new InstrumenterClassWriter(ClassWriter.COMPUTE_FRAMES, finder);
            final AsmCodeGenerator codeGenerator =
                    new AsmCodeGenerator(rootContainer, finder, this, false, classWriter);
            codeGenerator.patchFile(classFile);
            final FormErrorInfo[] warnings = codeGenerator.getWarnings();

            for (FormErrorInfo warning : warnings) {
                err.println(form.getAbsolutePath() + ": " + warning.getErrorMessage());
            }
            final FormErrorInfo[] errors = codeGenerator.getErrors();
            for (FormErrorInfo error : errors) {
                err.println(form.getAbsolutePath() + ": " + error.getErrorMessage());
            }
            if (errors.length > 0) {
                return false;
            }
        }
        return true;
    }

    /**
     * Takes {@code className} of the form "a.b.c.d", and returns one of the form "a/b/c$d" for
     * the first .class file found, or {@code null} otherwise.
     */
    private String getClassName(String className) {
        String[] split = className.split("\\.");
        for (int i = split.length - 1; i >= 0; i--) {
            String candidate = split[0];
            for (int j = 1; j < split.length; j++) {
                candidate += (j > i ? "$" : "/") + split[j];
            }
            File file = new File(mOutDir, candidate + ".class");
            if (file.exists()) {
                return candidate;
            }
        }
        return null;
    }

    private File getClassFile(String name) {
        name = getClassName(name);
        return name == null ? null : new File(mOutDir, name + ".class");
    }

    @Override
    public LwRootContainer loadForm(String formFilePath) throws Exception {
        if (mCache.containsKey(formFilePath)) {
            return mCache.get(formFilePath);
        }

        InputStream stream = null;
        String lowerFormFilePath = formFilePath.toLowerCase();
        // Try to find the nested form:
        File other = mOtherForms.get(lowerFormFilePath);
        if (other != null) {
            stream = new FileInputStream(other);
        }

        if (stream == null) {
            for (File file : mForms) {
                String name = file.getAbsolutePath().replace(File.separatorChar, '/').toLowerCase();
                if (name.endsWith(lowerFormFilePath)) {
                    stream = new FileInputStream(file);
                    break;
                }
            }
        }

        if (stream == null) {
            stream = mFinder.getLoader().getResourceAsStream(formFilePath);
        }

        if (stream != null) {
            // Stream is closed by SAXParser.
            final LwRootContainer container = Utils.getRootContainer(stream, null);
            mCache.put(formFilePath, container);
            return container;
        } else {
            throw new Exception("Cannot find nested form: " + formFilePath);
        }
    }

    @Override
    public String getClassToBindName(LwRootContainer container) {
        final String className = container.getClassToBind();
        String result = getClassName(className);
        return result != null ? result.replace('/', '.') : className;
    }
}
