/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.bazel.avd;

import org.junit.rules.ExternalResource;

/** JUnit rule that calls the executable generated by the avd rule to start and stop an emulator. */
public final class Emulator extends ExternalResource {

    private final String scriptPath;
    private final String binaryPath;
    private final String port;

    /**
     * Instantiates a rule that starts and stops an emulator created by the avd rule.
     *
     * @param scriptPath Path to the device executable generated by the avd rule.
     * @param port ADB port to start the emulator on.
     */
    public Emulator(String scriptPath, int port) {
        this(scriptPath,
                String.format("prebuilts/studio/sdk/%s/%s/emulator",
                        getOsDirName(),
                        getEmulatorDirName()),
                port);
    }

    /**
     * Instantiates a rule that starts and stops an emulator created by the avd rule.
     *
     * @param scriptPath Path to the script generated by the avd rule.
     * @param binaryPath Path to the actual emulator binary run by the script.
     * @param port ADB port to start the emulator on.
     */
    public Emulator(String scriptPath, String binaryPath, int port) {
        this.scriptPath = scriptPath;
        this.binaryPath = binaryPath;
        this.port = String.valueOf(port);
    }

    @Override
    public void before() throws Throwable {
        System.out.println("Starting emulator");
        exec(scriptPath, binaryPath, port);
        System.out.println("Emulator started");
    }

    @Override
    public void after() {
        try {
            System.out.println("Killing emulator");
            exec(scriptPath, binaryPath, "kill", port);
            System.out.println("Emulator killed");
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    private static void exec(String... cmd) throws Exception {
        int exitCode = new ProcessBuilder().command(cmd).inheritIO().start().waitFor();

        if (exitCode != 0) {
            throw new Exception(String.format("Emulator script exited with code: %d", exitCode));
        }
    }

    // Visible to be accessible by test.
    public static String getOsDirName() {
        String os = System.getProperty("os.name");
        if (os.startsWith("Mac OS")) {
            return "darwin";
        } else if (os.startsWith("Windows")) {
            return "windows";
        } else if (os.startsWith("Linux")) {
            return "linux";
        } else {
            throw new RuntimeException("Unsupported os.name: " + os);
        }
    }

    private static String getEmulatorDirName() {
        String os = System.getProperty("os.name");
        String arch = System.getProperty("os.arch");
        if (os.startsWith("Mac OS") && ("arm64".equals(arch) || "aarch64".equals(arch))) {
            return "emulator-arm64";
        } else {
            return "emulator";
        }
    }
}
