/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.lint.checks;

import com.android.tools.lint.checks.infrastructure.TestFile;
import com.android.tools.lint.detector.api.Detector;

public class PxUsageDetectorTest extends AbstractCheckTest {
    @Override
    protected Detector getDetector() {
        return new PxUsageDetector();
    }

    public void testPxWarnings() {
        String expected =
                ""
                        + "res/layout/now_playing_after.xml:49: Warning: Avoid using \"mm\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "        android:layout_width=\"100mm\"\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "res/layout/now_playing_after.xml:50: Warning: Avoid using \"in\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "        android:layout_height=\"120in\"\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "res/layout/now_playing_after.xml:41: Warning: Avoid using \"px\" as units; use \"dp\" instead [PxUsage]\n"
                        + "        android:layout_width=\"2px\"\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "0 errors, 3 warnings\n";
        //noinspection all // Sample code
        lint().files(
                        xml(
                                "res/layout/now_playing_after.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    android:id=\"@+id/now_playing\"\n"
                                        + "    android:layout_width=\"fill_parent\"\n"
                                        + "    android:layout_height=\"@dimen/now_playing_height\"\n"
                                        + "    android:orientation=\"horizontal\">\n"
                                        + "    <LinearLayout\n"
                                        + "        android:background=\"@color/background2\"\n"
                                        + "        android:paddingLeft=\"14dip\"\n"
                                        + "        android:paddingRight=\"14dip\"\n"
                                        + "        android:paddingTop=\"10dip\"\n"
                                        + "        android:paddingBottom=\"10dip\"\n"
                                        + "        android:layout_width=\"fill_parent\"\n"
                                        + "        android:layout_height=\"fill_parent\"\n"
                                        + "        android:layout_weight=\"1\"\n"
                                        + "        android:orientation=\"vertical\">\n"
                                        + "        <TextView\n"
                                        + "            android:id=\"@+id/now_playing_title\"\n"
                                        + "            android:duplicateParentState=\"true\"\n"
                                        + "            android:layout_width=\"fill_parent\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:textStyle=\"bold\"\n"
                                        + "            android:textSize=\"@dimen/text_size_large\"\n"
                                        + "            android:textColor=\"@color/foreground1\"\n"
                                        + "            android:text=\"@string/now_playing_after_title\"\n"
                                        + "            android:maxLines=\"2\"\n"
                                        + "            android:ellipsize=\"end\" />\n"
                                        + "        <TextView\n"
                                        + "            android:id=\"@+id/now_playing_subtitle\"\n"
                                        + "            android:duplicateParentState=\"true\"\n"
                                        + "            android:layout_width=\"fill_parent\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:paddingTop=\"3dip\"\n"
                                        + "            android:textColor=\"@color/foreground2\"\n"
                                        + "            android:textSize=\"@dimen/text_size_small\"\n"
                                        + "            android:text=\"@string/now_playing_after_subtitle\"\n"
                                        + "            android:singleLine=\"true\"\n"
                                        + "            android:ellipsize=\"end\" />\n"
                                        + "    </LinearLayout>\n"
                                        + "    <View\n"
                                        + "        android:layout_width=\"2px\"\n"
                                        + "        android:layout_height=\"fill_parent\"\n"
                                        + "        android:background=\"@android:color/white\" />\n"
                                        + "    <ImageButton\n"
                                        + "        android:background=\"@drawable/btn_now_playing_more\"\n"
                                        + "        android:id=\"@+id/now_playing_more\"\n"
                                        + "        android:src=\"@drawable/ic_now_playing_logo\"\n"
                                        + "        android:padding=\"12dip\"\n"
                                        + "        android:layout_width=\"100mm\"\n"
                                        + "        android:layout_height=\"120in\"\n"
                                        + "        android:onClick=\"onNowPlayingLogoClick\"\n"
                                        + "        android:maxHeight=\"1px\"\n"
                                        + "        android:scaleType=\"center\" />\n"
                                        + "</LinearLayout>\n"))
                .run()
                .expect(expected);
    }

    public void testSpWarnings() {
        String expected =
                ""
                        + "res/layout/textsize.xml:11: Warning: Should use \"sp\" instead of \"dp\" for text sizes [SpUsage]\n"
                        + "        android:textSize=\"14dp\" />\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "res/layout/textsize.xml:16: Warning: Should use \"sp\" instead of \"dp\" for text sizes [SpUsage]\n"
                        + "        android:textSize=\"14dip\" />\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "res/layout/textsize.xml:33: Warning: Avoid using sizes smaller than 11sp: 10sp [SmallSp]\n"
                        + "        android:textSize=\"10sp\" />\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "res/layout/textsize.xml:37: Warning: Avoid using sizes smaller than 11sp: 6.5sp [SmallSp]\n"
                        + "        android:layout_height=\"6.5sp\" />\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "0 errors, 4 warnings";
        //noinspection all // Sample code
        lint().files(
                        xml(
                                "res/layout/textsize.xml",
                                ""
                                        + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    xmlns:tools=\"http://schemas.android.com/tools\"\n"
                                        + "    android:id=\"@+id/LinearLayout1\"\n"
                                        + "    android:layout_width=\"match_parent\"\n"
                                        + "    android:layout_height=\"match_parent\"\n"
                                        + "    android:orientation=\"vertical\" >\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:textSize=\"14dp\" />\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:textSize=\"14dip\" />\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:textSize=\"14sp\" />\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:textSize=\"@android/dimen/mysizedp\" />\n"
                                        + "\n"
                                        + "    <!-- Small -->\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:textSize=\"10sp\" />\n"
                                        + "\n"
                                        + "    <ImageView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"6.5sp\" />\n"
                                        + "\n"
                                        + "    <!-- No warnings: wrong attribute, size == 0, etc -->\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:textSize=\"0sp\" />\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:marginTop=\"5sp\" />\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        // 11 is ok
                                        + "        android:textSize=\"11sp\" />\n"
                                        + "\n"
                                        + "</LinearLayout>\n"))
                .run()
                .expect(expected)
                .expectFixDiffs(
                        ""
                                + "Fix for res/layout/textsize.xml line 10: Replace with sp:\n"
                                + "@@ -11 +11\n"
                                + "-         android:textSize=\"14dp\" />\n"
                                + "+         android:textSize=\"14sp\" />\n"
                                + "Fix for res/layout/textsize.xml line 15: Replace with sp:\n"
                                + "@@ -16 +16\n"
                                + "-         android:textSize=\"14dip\" />\n"
                                + "+         android:textSize=\"14sp\" />\n");
    }

    public void testStyles() {
        String expected =
                ""
                        + "res/values/pxsp.xml:23: Warning: Avoid using \"mm\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "        <item name=\"android:textSize\">50mm</item>\n"
                        + "                                      ~~~~\n"
                        + "res/values/pxsp.xml:25: Warning: Avoid using \"in\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "            50in\n"
                        + "            ~~~~\n"
                        + "res/values/pxsp.xml:6: Warning: Should use \"sp\" instead of \"dp\" for text sizes [SpUsage]\n"
                        + "        <item name=\"android:textSize\">50dp</item>\n"
                        + "                                      ~~~~\n"
                        + "res/values/pxsp.xml:12: Warning: Should use \"sp\" instead of \"dp\" for text sizes [SpUsage]\n"
                        + "        <item name=\"android:textSize\"> 50dip </item>\n"
                        + "                                       ~~~~~\n"
                        + "res/values/pxsp.xml:9: Warning: Avoid using \"px\" as units; use \"dp\" instead [PxUsage]\n"
                        + "        <item name=\"android:textSize\">50px</item>\n"
                        + "                                      ~~~~\n"
                        + "res/values/pxsp.xml:17: Warning: Avoid using \"px\" as units; use \"dp\" instead [PxUsage]\n"
                        + "        <item name=\"android:paddingRight\"> 50px </item>\n"
                        + "                                           ~~~~\n"
                        + "res/values/pxsp.xml:18: Warning: Avoid using \"px\" as units; use \"dp\" instead [PxUsage]\n"
                        + "        <item name=\"android:paddingTop\">50px</item>\n"
                        + "                                        ~~~~\n"
                        + "0 errors, 7 warnings\n";
        //noinspection all // Sample code
        lint().files(
                        xml(
                                "res/values/pxsp.xml",
                                ""
                                        + "<resources>\n"
                                        + "    <style name=\"Style1\">\n"
                                        + "        <item name=\"android:textSize\">50sp</item>\n"
                                        + "    </style>\n"
                                        + "    <style name=\"Style2\">\n"
                                        + "        <item name=\"android:textSize\">50dp</item>\n"
                                        + "    </style>\n"
                                        + "    <style name=\"Style3\">\n"
                                        + "        <item name=\"android:textSize\">50px</item>\n"
                                        + "    </style>\n"
                                        + "    <style name=\"Style4\">\n"
                                        + "        <item name=\"android:textSize\"> 50dip </item>\n"
                                        + "    </style>\n"
                                        + " \n"
                                        + "    <style name=\"Style5\">\n"
                                        + "        <item name=\"android:paddingLeft\">@dimen/whats_on_item_padding</item>\n"
                                        + "        <item name=\"android:paddingRight\"> 50px </item>\n"
                                        + "        <item name=\"android:paddingTop\">50px</item>\n"
                                        + "        <item name=\"android:paddingBottom\">50dip</item>\n"
                                        + "    </style>\n"
                                        + "\n"
                                        + "    <style name=\"Style6\">\n"
                                        + "        <item name=\"android:textSize\">50mm</item>\n"
                                        + "        <item name=\"android:textSize\">\n"
                                        + "            50in\n"
                                        + "        </item>\n"
                                        + "    </style>\n"
                                        + "\n"
                                        + "    <style name=\"Widget.TabStrip\" parent=\"Widget\">\n"
                                        + "        <item name=\"android:divider\">?android:attr/listDivider</item>\n"
                                        + "        <item name=\"android:showDividers\">middle</item>\n"
                                        + "        <item name=\"android:dividerPadding\">0px</item>\n"
                                        + "        <item name=\"android:maxHeight\">1px</item>\n"
                                        + "    </style>\n"
                                        + "</resources>\n"))
                .run()
                .expect(expected)
                .verifyFixes()
                .expectFixDiffs(
                        ""
                                + "Fix for res/values/pxsp.xml line 5: Replace with sp:\n"
                                + "@@ -6 +6\n"
                                + "-         <item name=\"android:textSize\">50dp</item>\n"
                                + "+         <item name=\"android:textSize\">50sp</item>\n"
                                + "Fix for res/values/pxsp.xml line 11: Replace with sp:\n"
                                + "@@ -12 +12\n"
                                + "-         <item name=\"android:textSize\"> 50dip </item>\n"
                                + "+         <item name=\"android:textSize\"> 50sp </item>\n");
    }

    public void testFloatingPoint() {
        String expected =
                ""
                        + "res/values/dimens.xml:2: Warning: Avoid using \"in\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "    <item name=\"using_in_whole\" type=\"dimen\">5in</item>\n"
                        + "                                             ~~~\n"
                        + "res/values/dimens.xml:3: Warning: Avoid using \"in\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "    <item name=\"using_in\" type=\"dimen\">0.05in</item>\n"
                        + "                                       ~~~~~~\n"
                        + "res/layout/test.xml:3: Warning: Avoid using \"in\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "    android:layout_width=\"0.05in\"\n"
                        + "    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "res/layout/test.xml:4: Warning: Avoid using \"in\" as units (it does not work accurately on all devices); use \"dp\" instead [InOrMmUsage]\n"
                        + "    android:layout_height=\"5in\"\n"
                        + "    ~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "0 errors, 4 warnings";
        lint().files(
                        xml(
                                "res/values/dimens.xml",
                                ""
                                        + "<resources>\n"
                                        + "    <item name=\"using_in_whole\" type=\"dimen\">5in</item>\n"
                                        + "    <item name=\"using_in\" type=\"dimen\">0.05in</item>\n"
                                        + "</resources>"),
                        xml(
                                "res/layout/test.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    android:layout_width=\"0.05in\"\n"
                                        + "    android:layout_height=\"5in\"\n"
                                        + "    android:orientation=\"horizontal\">\n"
                                        + "</LinearLayout>\n"))
                .run()
                .expect(expected);
    }

    public void testIncrementalDimensions() {
        // Like testBatchDimensions, but because we specify a single file we're skipping
        // computing secondary locations
        String expected =
                ""
                        + "res/layout/textsize2.xml:9: Warning: Should use \"sp\" instead of \"dp\" for text sizes (@dimen/bottom_bar_portrait_button_font_size is defined as 16dp in values/dimens.xml [SpUsage]\n"
                        + "        android:textSize=\"@dimen/bottom_bar_portrait_button_font_size\"\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "0 errors, 1 warnings\n";
        lint().files(dimens, textsize2)
                .incremental("res/layout/textsize2.xml")
                .run()
                .expect(expected);
    }

    public void testBatchDimensions() {
        String expected =
                ""
                        + "res/layout/textsize2.xml:9: Warning: Should use \"sp\" instead of \"dp\" for text sizes (@dimen/bottom_bar_portrait_button_font_size is defined as 16dp in values/dimens.xml [SpUsage]\n"
                        + "        android:textSize=\"@dimen/bottom_bar_portrait_button_font_size\"\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "    res/values/dimens.xml:2: This dp dimension is used as a text size\n"
                        + "    <dimen name=\"bottom_bar_portrait_button_font_size\">16dp</dimen>\n"
                        + "                                                       ~~~~\n"
                        + "0 errors, 1 warnings\n";
        lint().files(dimens, textsize2).run().expect(expected);
    }

    public void test80029760() {
        // 80029760: SpUsage suppression on TextView doesn't make it go away from results
        lint().files(
                        xml(
                                "res/values/dimens.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<resources>\n"
                                        + "        <!-- http://www.androiddocs.com/design/style/iconography.html#small-contextual\n"
                                        + "             Within the body of your app, use small icons to surface actions and/or provide status for specific items. -->\n"
                                        + "        <dimen name=\"icon_context\">16dp</dimen>\n"
                                        + "</resources>\n"),
                        xml(
                                "res/layout/textsize2.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<LinearLayout\n"
                                        + "        xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "        xmlns:tools=\"http://schemas.android.com/tools\"\n"
                                        + "        android:layout_width=\"match_parent\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:orientation=\"horizontal\"\n"
                                        + "        tools:layout_height=\"24dp\"\n"
                                        + "        >\n"
                                        + "        <ImageView\n"
                                        + "                android:id=\"@+id/image\"\n"
                                        + "                android:layout_width=\"@dimen/icon_context\"\n"
                                        + "                android:layout_height=\"@dimen/icon_context\"\n"
                                        + "                android:layout_gravity=\"center_vertical\"\n"
                                        + "                android:contentDescription=\"@null\"\n"
                                        + "                tools:src=\"@android:drawable/sym_def_app_icon\"\n"
                                        + "                />\n"
                                        + "        <TextView\n"
                                        + "                android:id=\"@+id/type\"\n"
                                        + "                android:layout_width=\"@dimen/icon_context\"\n"
                                        + "                android:layout_height=\"match_parent\"\n"
                                        + "                android:gravity=\"center\"\n"
                                        + "                android:includeFontPadding=\"false\"\n"
                                        + "                android:textIsSelectable=\"false\"\n"
                                        + "                android:textSize=\"@dimen/icon_context\"\n"
                                        + "                android:typeface=\"monospace\"\n"
                                        + "                tools:text=\"+\"\n"
                                        + "                tools:ignore=\"SpUsage\"\n"
                                        + "                />\n"
                                        + "        <TextView\n"
                                        + "                android:id=\"@+id/title\"\n"
                                        + "                android:layout_width=\"wrap_content\"\n"
                                        + "                android:layout_height=\"match_parent\"\n"
                                        + "                android:gravity=\"center_vertical\"\n"
                                        + "                android:textIsSelectable=\"false\"\n"
                                        + "                tools:text=\"Category Name\"\n"
                                        + "                />\n"
                                        + "</LinearLayout>\n"))
                .run()
                .expectClean();
    }

    public void testSuppressDimension() {
        // 268575580: Allow suppressing on the dimension element instead
        lint().files(
                        xml(
                                "res/values/dimens.xml",
                                ""
                                        + "\n"
                                        + "<resources"
                                        + "    xmlns:tools=\"http://schemas.android.com/tools\"\n"
                                        + ">\n"
                                        + "    <dimen name=\"bottom_bar_portrait_button_font_size\" tools:ignore=\"SpUsage\">16dp</dimen>\n"
                                        + "</resources>\n"),
                        xml(
                                "res/layout/textsize2.xml",
                                ""
                                        + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    android:id=\"@+id/LinearLayout1\"\n"
                                        + "    android:orientation=\"vertical\" >\n"
                                        + "\n"
                                        + "    <TextView\n"
                                        + "        android:textSize=\"@dimen/bottom_bar_portrait_button_font_size\"\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\" />\n"
                                        + "\n"
                                        + "</LinearLayout>"))
                .run()
                .expectClean();
    }

    public void testIsZero() {
        assertFalse(PxUsageDetector.isZero(""));
        assertTrue(PxUsageDetector.isZero("0"));
        assertTrue(PxUsageDetector.isZero("0px"));
        assertTrue(PxUsageDetector.isZero("0.0"));
        assertTrue(PxUsageDetector.isZero("0.00"));
        assertTrue(PxUsageDetector.isZero("0.00px"));
        assertFalse(PxUsageDetector.isZero("1"));
        assertFalse(PxUsageDetector.isZero("1px"));
        assertFalse(PxUsageDetector.isZero("0.001px"));
    }

    @SuppressWarnings("all") // Sample code
    private TestFile dimens =
            xml(
                    "res/values/dimens.xml",
                    ""
                            + "<resources>\n"
                            + "    <dimen name=\"bottom_bar_portrait_button_font_size\">16dp</dimen>\n"
                            + "</resources>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile textsize2 =
            xml(
                    "res/layout/textsize2.xml",
                    ""
                            + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    xmlns:tools=\"http://schemas.android.com/tools\"\n"
                            + "    android:id=\"@+id/LinearLayout1\"\n"
                            + "    android:layout_width=\"match_parent\"\n"
                            + "    android:layout_height=\"match_parent\"\n"
                            + "    android:orientation=\"vertical\" >\n"
                            + "\n"
                            + "    <TextView\n"
                            + "        android:textSize=\"@dimen/bottom_bar_portrait_button_font_size\"\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\" />\n"
                            + "\n"
                            + "</LinearLayout>\n");
}
