/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.lint.checks;

import static com.android.tools.lint.checks.infrastructure.TestFiles.rClass;

import com.android.tools.lint.checks.infrastructure.TestMode;
import com.android.tools.lint.detector.api.Detector;

@SuppressWarnings({
    "EnhancedSwitchMigration",
    "ConcatenationWithEmptyString",
    "TextBlockMigration",
    "SwitchStatementWithTooFewBranches"
})
public class CutPasteDetectorTest extends AbstractCheckTest {
    @Override
    protected Detector getDetector() {
        return new CutPasteDetector();
    }

    public void testDocumentationExample() {
        lint().files(
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "import android.app.Activity;\n"
                                        + "import android.widget.Button;\n"
                                        + "\n"
                                        + "public class PasteError extends Activity {\n"
                                        + "    private Button next;\n"
                                        + "    private Button previous;\n"
                                        + "    protected void onCreate() {\n"
                                        + "        previous = (Button) findViewById(R.id.previous);\n"
                                        + "        next = (Button) findViewById(R.id.previous); // TYPO, meant R.id.next\n"
                                        + "    }\n"
                                        + "}"),
                        rClass("test.pkg", "@id/previous", "@id/next"))
                .run()
                .expect(
                        ""
                                + "src/test/pkg/PasteError.java:10: Warning: The id R.id.previous has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "        next = (Button) findViewById(R.id.previous); // TYPO, meant R.id.next\n"
                                + "                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:9: First usage here\n"
                                + "        previous = (Button) findViewById(R.id.previous);\n"
                                + "                            ~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "0 errors, 1 warnings");
    }

    public void testBasicJava() {

        //noinspection all // Sample code
        lint().files(
                        java(
                                "src/test/pkg/PasteError.java",
                                ""
                                        + "package test.pkg;\n"
                                        + "\n"
                                        + "import android.annotation.SuppressLint;\n"
                                        + "import android.app.Activity;\n"
                                        + "import android.view.LayoutInflater;\n"
                                        + "import android.view.View;\n"
                                        + "import android.view.ViewGroup;\n"
                                        + "import android.widget.Button;\n"
                                        + "import android.widget.TextView;\n"
                                        + "\n"
                                        + "@SuppressWarnings({\"ConstantConditions\", \"UnnecessaryLocalVariable\", \"ConstantIfStatement\",\n"
                                        + "        \"StatementWithEmptyBody\", \"FieldCanBeLocal\", \"unused\", \"UnusedAssignment\"})\n"
                                        + "public class PasteError extends Activity {\n"
                                        + "    protected void ok() {\n"
                                        + "        Button button1 = (Button) findViewById(R.id.textView1); // OK\n"
                                        + "        mView2 = findViewById(R.id.textView2); // OK\n"
                                        + "        View view3 = findViewById(R.id.activity_main); // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void error() {\n"
                                        + "        View view1 = findViewById(R.id.duplicated);\n"
                                        + "        View view2 = findViewById(R.id.duplicated); // ERROR 1\n"
                                        + "        View view3 = findViewById(R.id.ok); // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void ok2() {\n"
                                        + "        View view1;\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById(R.id.ok);\n"
                                        + "        } else {\n"
                                        + "            view1 = findViewById(R.id.ok); // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    @SuppressLint(\"CutPasteId\")\n"
                                        + "    protected void suppressed() {\n"
                                        + "        View view1 = findViewById(R.id.duplicated);\n"
                                        + "        View view2 = findViewById(R.id.duplicated); // OK (because suppressed)\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    private void ok3() {\n"
                                        + "        if (view == null || view.findViewById(R.id.city_name) == null) {\n"
                                        + "            view = mInflater.inflate(R.layout.city_list_item, parent, false);\n"
                                        + "        }\n"
                                        + "        TextView name = (TextView) view.findViewById(R.id.city_name); // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    private void ok4() {\n"
                                        + "        mPrevAlbumWrapper = mPrevTrackLayout.findViewById(R.id.album_wrapper);\n"
                                        + "        mNextAlbumWrapper = mNextTrackLayout.findViewById(R.id.album_wrapper); // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    public View getView(int position, View convertView, ViewGroup parent) {\n"
                                        + "        View listItem = convertView;\n"
                                        + "        if (getItemViewType(position) == VIEW_TYPE_HEADER) {\n"
                                        + "            TextView header = (TextView) listItem.findViewById(R.id.name);\n"
                                        + "        } else if (getItemViewType(position) == VIEW_TYPE_BOOLEAN) {\n"
                                        + "            TextView filterName = (TextView) listItem.findViewById(R.id.name); // OK\n"
                                        + "        } else {\n"
                                        + "            TextView filterName = (TextView) listItem.findViewById(R.id.name); // OK\n"
                                        + "        }\n"
                                        + "        return null;\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void ok_branch_1() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById(R.id.ok);\n"
                                        + "        } else {\n"
                                        + "            view2 = findViewById(R.id.ok); // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void error_branch_1() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById(R.id.duplicated);\n"
                                        + "        }\n"
                                        + "        if (true) {\n"
                                        + "            view2 = findViewById(R.id.duplicated); // ERROR 2\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void error_branch_2() {\n"
                                        + "        view1 = findViewById(R.id.duplicated);\n"
                                        + "        if (true) {\n"
                                        + "            view2 = findViewById(R.id.duplicated); // ERROR 3\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void error_branch_3() {\n"
                                        + "        view1 = findViewById(R.id.duplicated);\n"
                                        + "        if (true) {\n"
                                        + "        } else {\n"
                                        + "            view2 = findViewById(R.id.duplicated); // ERROR 4\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void error_branch_4() {\n"
                                        + "        view1 = findViewById(R.id.duplicated);\n"
                                        + "        if (true) {\n"
                                        + "        } else {\n"
                                        + "            if (true) {\n"
                                        + "                view2 = findViewById(R.id.duplicated); // ERROR 5\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void ok_branch_2() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById(R.id.ok);\n"
                                        + "        } else {\n"
                                        + "            if (true) {\n"
                                        + "                view2 = findViewById(R.id.ok); // OK\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void ok_branch3() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById(R.id.ok);\n"
                                        + "            return;\n"
                                        + "        }\n"
                                        + "        if (true) {\n"
                                        + "            view2 = findViewById(R.id.ok); // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    public static void ok_switch(View root, int position) {\n"
                                        + "        // mutually exclusive branches\n"
                                        + "        switch (position) {\n"
                                        + "            case 0: {\n"
                                        + "                TextView titleView = (TextView) root.findViewById(R.id.ok);\n"
                                        + "            }\n"
                                        + "            break;\n"
                                        + "            default: {\n"
                                        + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.ok); // OK\n"
                                        + "            }\n"
                                        + "            break;\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    public static void error_switch_fallthrough(View root, int position) {\n"
                                        + "        switch (position) {\n"
                                        + "            case 0: {\n"
                                        + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                        + "                // fallthrough!\n"
                                        + "            }\n"
                                        + "            default: {\n"
                                        + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 6\n"
                                        + "            }\n"
                                        + "            break;\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    public static void warning_switch_to_outer(View root, int position) {\n"
                                        + "        switch (position) {\n"
                                        + "            case 0:\n"
                                        + "            {\n"
                                        + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                        + "            }\n"
                                        + "            break;\n"
                                        + "        }\n"
                                        + "        TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 7\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    public void while_loop_error(View root, int position) {\n"
                                        + "        while (position-- > 0) { // here we can flow back\n"
                                        + "            if (true) {\n"
                                        + "                view1 = findViewById(R.id.duplicated);\n"
                                        + "            } else {\n"
                                        + "                if (true) {\n"
                                        + "                    view2 = findViewById(R.id.duplicated); // ERROR 8\n"
                                        + "                }\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected void require_by_id_error_branch_1() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = requireViewById(R.id.duplicated);\n"
                                        + "        }\n"
                                        + "        if (true) {\n"
                                        + "            view2 = requireViewById(R.id.duplicated); // ERROR 9\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    public final <T extends View> T requireViewById(int id) {\n"
                                        + "        throw new RuntimeException(\"Stub!\");\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    private View view1;\n"
                                        + "    private View mView2;\n"
                                        + "    private View view;\n"
                                        + "    private View view2;\n"
                                        + "    private LayoutInflater mInflater;\n"
                                        + "    private Object mPrevAlbumWrapper;\n"
                                        + "    private Object mNextAlbumWrapper;\n"
                                        + "    private Activity mPrevTrackLayout;\n"
                                        + "    private Activity mNextTrackLayout;\n"
                                        + "    private android.view.ViewGroup parent;\n"
                                        + "    private static final int VIEW_TYPE_HEADER = 1;\n"
                                        + "    private static final int VIEW_TYPE_BOOLEAN = 2;\n"
                                        + "    private int getItemViewType(int position) {\n"
                                        + "        return VIEW_TYPE_BOOLEAN;\n"
                                        + "    }\n"
                                        + "}"),
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "\n"
                                        + "public final class R {\n"
                                        + "    public static final class id {\n"
                                        + "        public static final int ok = 0x7f0a0000;\n"
                                        + "        public static final int duplicated = 0x7f0a0000;\n"
                                        + "        public static final int textView1 = 0x7f0a0001;\n"
                                        + "        public static final int textView2 = 0x7f0a0002;\n"
                                        + "        public static final int activity_main = 0x7f0a0003;\n"
                                        + "        public static final int album_wrapper = 0x7f0a0004;\n"
                                        + "        public static final int city_name = 0x7f0a0005;\n"
                                        + "        public static final int name = 0x7f0a0006;\n"
                                        + "    }\n"
                                        + "    public static final class layout {\n"
                                        + "        public static final int city_list_item = 0x7f0b0000;\n"
                                        + "    }\n"
                                        + "}\n"))
                .run()
                .expect(
                        ""
                                + "src/test/pkg/PasteError.java:22: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "        View view2 = findViewById(R.id.duplicated); // ERROR 1\n"
                                + "                     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:21: First usage here\n"
                                + "        View view1 = findViewById(R.id.duplicated);\n"
                                + "                     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:78: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = findViewById(R.id.duplicated); // ERROR 2\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:75: First usage here\n"
                                + "            view1 = findViewById(R.id.duplicated);\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:85: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = findViewById(R.id.duplicated); // ERROR 3\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:83: First usage here\n"
                                + "        view1 = findViewById(R.id.duplicated);\n"
                                + "                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:93: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = findViewById(R.id.duplicated); // ERROR 4\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:90: First usage here\n"
                                + "        view1 = findViewById(R.id.duplicated);\n"
                                + "                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:102: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                view2 = findViewById(R.id.duplicated); // ERROR 5\n"
                                + "                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:98: First usage here\n"
                                + "        view1 = findViewById(R.id.duplicated);\n"
                                + "                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:148: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 6\n"
                                + "                                                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:144: First usage here\n"
                                + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                + "                                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:162: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "        TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 7\n"
                                + "                                               ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:158: First usage here\n"
                                + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                + "                                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:171: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                    view2 = findViewById(R.id.duplicated); // ERROR 8\n"
                                + "                            ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:168: First usage here\n"
                                + "                view1 = findViewById(R.id.duplicated);\n"
                                + "                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/PasteError.java:182: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = requireViewById(R.id.duplicated); // ERROR 9\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/PasteError.java:179: First usage here\n"
                                + "            view1 = requireViewById(R.id.duplicated);\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "0 errors, 9 warnings");
    }

    public void testBasicKotlin() {
        // Like testBasicJava(), but for Kotlin. When statements are tested separately.
        lint().files(
                        kotlin(
                                // language=TEXT
                                ""
                                        + "package com.example.testapp\n"
                                        + "\n"
                                        + "import android.annotation.SuppressLint\n"
                                        + "import android.app.Activity\n"
                                        + "import android.view.LayoutInflater\n"
                                        + "import android.view.View\n"
                                        + "import android.view.ViewGroup\n"
                                        + "import android.widget.Button\n"
                                        + "import android.widget.TextView\n"
                                        + "\n"
                                        + "@Suppress(\"unused\")\n"
                                        + "class PasteError : Activity() {\n"
                                        + "    protected fun ok() {\n"
                                        + "        val button1 = findViewById<View>(R.id.textView1) as Button\n"
                                        + "        mView2 = findViewById<View>(R.id.textView2) // OK\n"
                                        + "        val view3 = findViewById<View>(R.id.activity_main) // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun error() {\n"
                                        + "        val view1 = findViewById<View>(R.id.duplicated)\n"
                                        + "        val view2 = findViewById<View>(R.id.duplicated) // ERROR 1\n"
                                        + "        val view3 = findViewById<View>(R.id.ok) // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun ok2() {\n"
                                        + "        val view1: View\n"
                                        + "        view1 = if (true) {\n"
                                        + "            findViewById<View>(R.id.ok)\n"
                                        + "        } else {\n"
                                        + "            findViewById<View>(R.id.ok) // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    @SuppressLint(\"CutPasteId\")\n"
                                        + "    protected fun suppressed() {\n"
                                        + "        val view1 = findViewById<View>(R.id.duplicated)\n"
                                        + "        val view2 = findViewById<View>(R.id.duplicated) // OK (because suppressed)\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    private fun ok3() {\n"
                                        + "        if (view == null || view!!.findViewById<View?>(R.id.city_name) == null) {\n"
                                        + "            view = mInflater.inflate(R.layout.city_list_item, parent, false)\n"
                                        + "        }\n"
                                        + "        val name = view?.findViewById<View>(R.id.city_name) as TextView // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    private fun ok4() {\n"
                                        + "        mPrevAlbumWrapper = mPrevTrackLayout!!.findViewById<View>(R.id.album_wrapper)\n"
                                        + "        mNextAlbumWrapper = mNextTrackLayout!!.findViewById<View>(R.id.album_wrapper) // OK\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    fun getView(position: Int, convertView: View, parent: ViewGroup?): View? {\n"
                                        + "        if (getItemViewType(position) == VIEW_TYPE_HEADER) {\n"
                                        + "            val header =\n"
                                        + "                convertView.findViewById<View>(R.id.name) as TextView\n"
                                        + "        } else if (getItemViewType(position) == VIEW_TYPE_BOOLEAN) {\n"
                                        + "            val filterName =\n"
                                        + "                convertView.findViewById<View>(R.id.name) as TextView // OK\n"
                                        + "        } else {\n"
                                        + "            val filterName =\n"
                                        + "                convertView.findViewById<View>(R.id.name) as TextView // OK\n"
                                        + "        }\n"
                                        + "        return null\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun ok_branch_1() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById<View>(R.id.ok)\n"
                                        + "        } else {\n"
                                        + "            view2 = findViewById<View>(R.id.ok) // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun error_branch_1() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById<View>(R.id.duplicated)\n"
                                        + "        }\n"
                                        + "        if (true) {\n"
                                        + "            view2 = findViewById<View>(R.id.duplicated) // ERROR 2\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun error_branch_2() {\n"
                                        + "        view1 = findViewById<View>(R.id.duplicated)\n"
                                        + "        if (true) {\n"
                                        + "            view2 = findViewById<View>(R.id.duplicated) // ERROR 3\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun error_branch_3() {\n"
                                        + "        view1 = findViewById<View>(R.id.duplicated)\n"
                                        + "        if (true) {\n"
                                        + "        } else {\n"
                                        + "            view2 = findViewById<View>(R.id.duplicated) // ERROR 4\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun error_branch_4() {\n"
                                        + "        view1 = findViewById<View>(R.id.duplicated)\n"
                                        + "        if (true) {\n"
                                        + "        } else {\n"
                                        + "            if (true) {\n"
                                        + "                view2 = findViewById<View>(R.id.duplicated) // ERROR 5\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun ok_branch_2() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById<View>(R.id.ok)\n"
                                        + "        } else {\n"
                                        + "            if (true) {\n"
                                        + "                view2 = findViewById<View>(R.id.ok) // OK\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun ok_branch3() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = findViewById<View>(R.id.ok)\n"
                                        + "            return\n"
                                        + "        }\n"
                                        + "        if (true) {\n"
                                        + "            view2 = findViewById<View>(R.id.ok) // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    fun while_loop_error(root: View?, position: Int) {\n"
                                        + "        var position = position\n"
                                        + "        while (position-- > 0) { // here we can flow back\n"
                                        + "            if (true) {\n"
                                        + "                view1 = findViewById<View>(R.id.duplicated)\n"
                                        + "            } else {\n"
                                        + "                if (true) {\n"
                                        + "                    view2 = findViewById<View>(R.id.duplicated) // ERROR 6\n"
                                        + "                }\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    protected fun require_by_id_error_branch_1() {\n"
                                        + "        if (true) {\n"
                                        + "            view1 = requireViewById<View>(R.id.duplicated)\n"
                                        + "        }\n"
                                        + "        if (true) {\n"
                                        + "            view2 = requireViewById<View>(R.id.duplicated) // ERROR 7\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    fun <T : View?> requireViewById(id: Int): T {\n"
                                        + "        throw RuntimeException(\"Stub!\")\n"
                                        + "    }\n"
                                        + "\n"
                                        + "    private var view1: View? = null\n"
                                        + "    private var mView2: View? = null\n"
                                        + "    private var view: View? = null\n"
                                        + "    private var view2: View? = null\n"
                                        + "    private val mInflater: LayoutInflater? = null\n"
                                        + "    private var mPrevAlbumWrapper: Any? = null\n"
                                        + "    private var mNextAlbumWrapper: Any? = null\n"
                                        + "    private val mPrevTrackLayout: Activity? = null\n"
                                        + "    private val mNextTrackLayout: Activity? = null\n"
                                        + "    private val parent: ViewGroup? = null\n"
                                        + "    private fun getItemViewType(position: Int): Int {\n"
                                        + "        return VIEW_TYPE_BOOLEAN\n"
                                        + "    }\n"
                                        + "}"
                                        + "\n"
                                        + "const val VIEW_TYPE_HEADER = 1\n"
                                        + "const val VIEW_TYPE_BOOLEAN = 2\n"
                                        + "// When statements are tested separately"),
                        java(
                                ""
                                        + "package com.example.testapp;\n"
                                        + "\n"
                                        + "public class R {\n"
                                        + "    static class id {\n"
                                        + "        public static final int textView1 = 1;\n"
                                        + "        public static final int textView2 = 1;\n"
                                        + "        public static final int activity_main = 1;\n"
                                        + "        public static final int duplicated = 1;\n"
                                        + "        public static final int ok = 1;\n"
                                        + "        public static final int city_name = 1;\n"
                                        + "        public static final int album_wrapper = 1;\n"
                                        + "        public static final int name = 1;\n"
                                        + "    }\n"
                                        + "    static class layout {\n"
                                        + "        public static final int city_list_item = 1;\n"
                                        + "    }\n"
                                        + "}\n"))
                .testModes(TestMode.DEFAULT)
                .run()
                .expect(
                        "\n"
                                + "src/com/example/testapp/PasteError.kt:21: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "        val view2 = findViewById<View>(R.id.duplicated) // ERROR 1\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/com/example/testapp/PasteError.kt:20: First usage here\n"
                                + "        val view1 = findViewById<View>(R.id.duplicated)\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/com/example/testapp/PasteError.kt:79: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = findViewById<View>(R.id.duplicated) // ERROR 2\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/com/example/testapp/PasteError.kt:76: First usage here\n"
                                + "            view1 = findViewById<View>(R.id.duplicated)\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/com/example/testapp/PasteError.kt:86: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = findViewById<View>(R.id.duplicated) // ERROR 3\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/com/example/testapp/PasteError.kt:84: First usage here\n"
                                + "        view1 = findViewById<View>(R.id.duplicated)\n"
                                + "                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/com/example/testapp/PasteError.kt:94: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = findViewById<View>(R.id.duplicated) // ERROR 4\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/com/example/testapp/PasteError.kt:91: First usage here\n"
                                + "        view1 = findViewById<View>(R.id.duplicated)\n"
                                + "                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/com/example/testapp/PasteError.kt:103: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                view2 = findViewById<View>(R.id.duplicated) // ERROR 5\n"
                                + "                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/com/example/testapp/PasteError.kt:99: First usage here\n"
                                + "        view1 = findViewById<View>(R.id.duplicated)\n"
                                + "                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/com/example/testapp/PasteError.kt:135: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                    view2 = findViewById<View>(R.id.duplicated) // ERROR 6\n"
                                + "                            ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/com/example/testapp/PasteError.kt:132: First usage here\n"
                                + "                view1 = findViewById<View>(R.id.duplicated)\n"
                                + "                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/com/example/testapp/PasteError.kt:146: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "            view2 = requireViewById<View>(R.id.duplicated) // ERROR 7\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/com/example/testapp/PasteError.kt:143: First usage here\n"
                                + "            view1 = requireViewById<View>(R.id.duplicated)\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "0 errors, 7 warnings");
    }

    public void testCompareFindViewById() {
        //noinspection all // Sample code
        lint().files(
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "import android.app.Activity;\n"
                                        + "import android.view.LayoutInflater;\n"
                                        + "import android.view.View;\n"
                                        + "\n"
                                        + "public class FindViewByIdTest extends Activity {\n"
                                        + "    private boolean mScrollingHeroView;\n"
                                        + "    private HeroViewHolder mHeroViewHolder;\n"
                                        + "\n"
                                        + "    void test() {\n"
                                        + "        mScrollingHeroView = findViewById(R.id.alarm_hero_view) == null;\n"
                                        + "\n"
                                        + "        if (mScrollingHeroView) {\n"
                                        + "            mHeroViewHolder = new HeroViewHolder(LayoutInflater.from(this)\n"
                                        + "                    .inflate(R.layout.alarm_card_hero_view, this, false));\n"
                                        + "        } else {\n"
                                        + "            mHeroViewHolder = new HeroViewHolder(findViewById(R.id.alarm_hero_view));\n"
                                        + "        }        \n"
                                        + "    }\n"
                                        + "\n"
                                        + "    private static class HeroViewHolder {\n"
                                        + "        public HeroViewHolder(View view) {\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "}\n"),
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "public final class R {\n"
                                        + "    public static final class id {\n"
                                        + "        public static final int alarm_hero_view = 0x7f0a0000;\n"
                                        + "    }\n"
                                        + "    public static final class layout {\n"
                                        + "        public static final int alarm_card_hero_view = 0x7f0b0000;\n"
                                        + "    }\n"
                                        + "}\n"
                                        + ""))
                .run()
                .expectClean();
    }

    public void testIdsUnresolvable() {
        //noinspection all // Sample code
        lint().files(
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "import android.app.Activity;\n"
                                        + "import android.view.View;\n"
                                        + "\n"
                                        + "public class FindViewByIdTest extends Activity {\n"
                                        + "    void test() {\n"
                                        + "        View view1 = findViewById(R.id.view1);\n"
                                        + "        View view2 = findViewById(R.id.view2);\n"
                                        + "    }\n"
                                        + "}\n"),
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "public final class R {\n"
                                        + "    public static final class id {}\n"
                                        + "}\n"
                                        + ""))
                .run()
                .expectClean();
    }

    public void testIdsUnresolvable_butStillWrong() {
        //noinspection all // Sample code
        lint().files(
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "import android.app.Activity;\n"
                                        + "import android.view.View;\n"
                                        + "\n"
                                        + "public class FindViewByIdTest extends Activity {\n"
                                        + "    void test() {\n"
                                        + "        View view1 = findViewById(test.pkg1.R.id.view1);\n"
                                        + "        View view2 = findViewById(test.pkg2.R.id.view1);\n"
                                        + "    }\n"
                                        + "}\n"),
                        java(
                                ""
                                        + "package test.pkg1;\n"
                                        + "public final class R {\n"
                                        + "    public static final class id {}\n"
                                        + "}\n"
                                        + ""),
                        java(
                                ""
                                        + "package test.pkg2;\n"
                                        + "public final class R {\n"
                                        + "    public static final class id {\n"
                                        + "        public static int view1;\n"
                                        + "        public static int view2;\n"
                                        + "    }\n"
                                        + "}\n"
                                        + ""))
                .run()
                .expectWarningCount(1);
    }

    public void testSwitchAndWhen() {
        lint().files(
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "\n"
                                        + "import android.view.View;\n"
                                        + "import android.widget.TextView;\n"
                                        + "\n"
                                        + "class Test {\n"
                                        + "    public static void no_warning_mutually_exclusive(View root, int position) {\n"
                                        + "        switch (position) {\n"
                                        + "            case 0: {\n"
                                        + "                TextView titleView = (TextView) root.findViewById(R.id.ok);\n"
                                        + "                break;\n"
                                        + "            }\n"
                                        + "            case 1: {\n"
                                        + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.ok); // OK\n"
                                        + "            }\n"
                                        + "            break;\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "    public static void error_fall_through1(View root, int position) {\n"
                                        + "        switch (position) {\n"
                                        + "            case 0: {\n"
                                        + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                        + "                // no break; fallthrough:\n"
                                        + "            }\n"
                                        + "            case 1: {\n"
                                        + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 1\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "\n"
                                        + "\n"
                                        + "    public static void error_fall_through2(View root, int position) {\n"
                                        + "        switch (position) {\n"
                                        + "            case 0: {\n"
                                        + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                        + "                // no break; fallthrough:\n"
                                        + "            }\n"
                                        + "            default: {\n"
                                        + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 2\n"
                                        + "            }\n"
                                        + "            break;\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "}"),
                        kotlin(
                                ""
                                        + "package test.pkg\n"
                                        + "\n"
                                        + "import android.view.View\n"
                                        + "import android.widget.TextView\n"
                                        + "\n"
                                        + "fun no_warning_mutually_exclusive(root: View, position: Int) {\n"
                                        + "    when (position) {\n"
                                        + "        0 -> {\n"
                                        + "            val titleView = root.findViewById<View>(R.id.ok) as TextView\n"
                                        + "            // no fall through in Kotlin\n"
                                        + "        }\n"
                                        + "\n"
                                        + "        1 -> {\n"
                                        + "            val sectionTitleView = root.findViewById<View>(R.id.ok) as TextView // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "}\n"
                                        + "\n"
                                        + "fun no_warning_mutually_exclusive_no_subject(root: View, position: Int) {\n"
                                        + "    when {\n"
                                        + "        position == 0 -> {\n"
                                        + "            val titleView = root.findViewById<View>(R.id.ok) as TextView\n"
                                        + "        }\n"
                                        + "\n"
                                        + "        position == 1 -> {\n"
                                        + "            val sectionTitleView = root.findViewById<View>(R.id.ok) as TextView // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "}\n"
                                        + "\n"
                                        + "fun error_switch_fallthrough(root: View, position: Int) {\n"
                                        + "    when (position) {\n"
                                        + "        0 -> {\n"
                                        + "            run {\n"
                                        + "                val titleView =\n"
                                        + "                    root.findViewById<View>(R.id.duplicated) as TextView\n"
                                        + "            }\n"
                                        + "            run {\n"
                                        + "                val sectionTitleView =\n"
                                        + "                    root.findViewById<View>(R.id.duplicated) as TextView // ERROR 3\n"
                                        + "            }\n"
                                        + "        }\n"
                                        + "\n"
                                        + "        else -> {\n"
                                        + "            val sectionTitleView = root.findViewById<View>(R.id.duplicated) as TextView // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "}\n"
                                        + "\n"
                                        + "fun testWhen(root: View, position: Int) {\n"
                                        + "    // mutually exclusive branches\n"
                                        + "    when (position) {\n"
                                        + "        0 -> {\n"
                                        + "            val titleView = root.findViewById<View>(R.id.ok) as TextView\n"
                                        + "        }\n"
                                        + "\n"
                                        + "        else -> {\n"
                                        + "            val sectionTitleView = root.findViewById<View>(R.id.ok) as TextView // OK\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "}\n"),
                        rClass("test.pkg", "@id/duplicated", "@id/ok"))
                .run()
                .expect(
                        ""
                                + "src/test/pkg/Test.java:26: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 1\n"
                                + "                                                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/Test.java:22: First usage here\n"
                                + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                + "                                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/Test.java:39: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 2\n"
                                + "                                                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/Test.java:35: First usage here\n"
                                + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                + "                                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/test.kt:40: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "                    root.findViewById<View>(R.id.duplicated) as TextView // ERROR 3\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/test.kt:36: First usage here\n"
                                + "                    root.findViewById<View>(R.id.duplicated) as TextView\n"
                                + "                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "0 errors, 3 warnings");
    }

    public void testSwitchFlowToOuter() {
        lint().files(
                        java(
                                ""
                                        + "package test.pkg;\n"
                                        + "\n"
                                        + "import android.view.View;\n"
                                        + "import android.widget.TextView;\n"
                                        + "\n"
                                        + "class Test {\n"
                                        + "    public static void error_switch_to_outer(View root, int position) {\n"
                                        + "        switch (position) {\n"
                                        + "            case 0:\n"
                                        + "            {\n"
                                        + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                        + "            }\n"
                                        + "            break;\n"
                                        + "        }\n"
                                        + "        TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 1\n"
                                        + "    }\n"
                                        + "}"),
                        kotlin(
                                ""
                                        + "package test.pkg\n"
                                        + "\n"
                                        + "import android.view.View\n"
                                        + "import android.widget.TextView\n"
                                        + "\n"
                                        + "fun error_switch_to_outer(root: View, position: Int) {\n"
                                        + "    when (position) {\n"
                                        + "        0 -> {\n"
                                        + "            val titleView = root.findViewById<View>(R.id.duplicated) as TextView\n"
                                        + "        }\n"
                                        + "    }\n"
                                        + "    val sectionTitleView = root.findViewById<View>(R.id.duplicated) as TextView // ERROR 2\n"
                                        + "}\n"),
                        rClass("test.pkg", "@id/duplicated"))
                .run()
                .expect(
                        ""
                                + "src/test/pkg/Test.java:15: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "        TextView sectionTitleView = (TextView) root.findViewById(R.id.duplicated); // ERROR 1\n"
                                + "                                               ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/Test.java:11: First usage here\n"
                                + "                TextView titleView = (TextView) root.findViewById(R.id.duplicated);\n"
                                + "                                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "src/test/pkg/test.kt:12: Warning: The id R.id.duplicated has already been looked up in this method; possible cut & paste error? [CutPasteId]\n"
                                + "    val sectionTitleView = root.findViewById<View>(R.id.duplicated) as TextView // ERROR 2\n"
                                + "                           ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "    src/test/pkg/test.kt:9: First usage here\n"
                                + "            val titleView = root.findViewById<View>(R.id.duplicated) as TextView\n"
                                + "                            ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                                + "0 errors, 2 warnings");
    }
}
