/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.lint.checks;

import com.android.tools.lint.checks.infrastructure.TestFile;
import com.android.tools.lint.detector.api.Detector;

@SuppressWarnings("javadoc")
public class ButtonDetectorTest extends AbstractCheckTest {
    @Override
    protected Detector getDetector() {
        return new ButtonDetector();
    }

    public void testButtonOrder() {
        //noinspection all // Sample code
        lint().files(manifest().minSdk(14), mButtonbar, mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expect(
                        ""
                                + "res/layout/buttonbar.xml:12: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:44: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:92: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:124: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:140: Warning: OK button should be on the right (was \"Ok | CANCEL\", should be \"CANCEL | Ok\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:156: Warning: OK button should be on the right (was \"OK | Abort\", should be \"Abort | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:177: Warning: Cancel button should be on the left (was \"Send | Cancel\", should be \"Cancel | Send\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 7 warnings\n");
    }

    public void testButtonOrder2() {
        lint().files(mMinsdk5targetsdk14, mButtonbar2, mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expect(
                        ""
                                + "res/layout-v14/buttonbar.xml:12: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v14/buttonbar.xml:44: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v14/buttonbar.xml:92: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v14/buttonbar.xml:124: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v14/buttonbar.xml:140: Warning: OK button should be on the right (was \"Ok | CANCEL\", should be \"CANCEL | Ok\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v14/buttonbar.xml:156: Warning: OK button should be on the right (was \"OK | Abort\", should be \"Abort | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v14/buttonbar.xml:177: Warning: Cancel button should be on the left (was \"Send | Cancel\", should be \"Cancel | Send\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 7 warnings\n");
    }

    public void testButtonOrder3() {
        // Similar to test 3, but also complain if the -v version is *higher* than 14
        lint().files(mMinsdk5targetsdk14, mButtonbar3, mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expect(
                        ""
                                + "res/layout-v16/buttonbar.xml:12: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v16/buttonbar.xml:44: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v16/buttonbar.xml:92: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v16/buttonbar.xml:124: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v16/buttonbar.xml:140: Warning: OK button should be on the right (was \"Ok | CANCEL\", should be \"CANCEL | Ok\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v16/buttonbar.xml:156: Warning: OK button should be on the right (was \"OK | Abort\", should be \"Abort | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-v16/buttonbar.xml:177: Warning: Cancel button should be on the left (was \"Send | Cancel\", should be \"Cancel | Send\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 7 warnings\n");
    }

    public void testButtonOrder4() {
        // Targeting 14 but using a layout that also needs to work for older platforms:
        lint().files(mMinsdk5targetsdk14, mButtonbar, mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expect(
                        ""
                                + "res/layout/buttonbar.xml:12: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:44: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:92: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:124: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:140: Warning: OK button should be on the right (was \"Ok | CANCEL\", should be \"CANCEL | Ok\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:156: Warning: OK button should be on the right (was \"OK | Abort\", should be \"Abort | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:177: Warning: Cancel button should be on the left (was \"Send | Cancel\", should be \"Cancel | Send\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 7 warnings\n");
    }

    public void testSuppressed() {
        lint().files(
                        manifest().minSdk(14),
                        xml(
                                "res/layout/buttonbar_suppressed.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    xmlns:tools=\"http://schemas.android.com/tools\"\n"
                                        + "    android:layout_width=\"match_parent\"\n"
                                        + "    android:layout_height=\"match_parent\"\n"
                                        + "    android:orientation=\"vertical\" >\n"
                                        + "\n"
                                        + "    <!-- Hardcoded strings, wrong order -->\n"
                                        + "    <LinearLayout\n"
                                        + "        android:layout_width=\"match_parent\"\n"
                                        + "        android:layout_height=\"wrap_content\" >\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"OK\"\n"
                                        + "            tools:ignore=\"ButtonOrder\" />\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"Cancel\"\n"
                                        + "            tools:ignore=\"ButtonOrder\" />\n"
                                        + "    </LinearLayout>\n"
                                        + "\n"
                                        + "    <!-- @android:string resources, wrong order -->\n"
                                        + "    <LinearLayout\n"
                                        + "        android:layout_width=\"match_parent\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        tools:ignore=\"ButtonOrder\" >\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"@android:string/ok\" />\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"@android:string/cancel\" />\n"
                                        + "    </LinearLayout>\n"
                                        + "\n"
                                        + "</LinearLayout>\n"),
                        mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expectClean();
    }

    public void testButtonOrderRelativeLayout() {
        lint().files(mButtonbar2_class, mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expectClean();
    }

    public void testButtonOrderRelativeLayout2() {
        lint().files(manifest().minSdk(14), mButtonbar3_class, mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expect(
                        ""
                                + "res/layout/buttonbar3.xml:27: Warning: Cancel button should be on the left [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 1 warnings\n");
    }

    public void testButtonOrderRelativeLayout3() {
        lint().files(
                        xml(
                                "res/layout/buttonbar4.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<RelativeLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    android:layout_width=\"match_parent\"\n"
                                        + "    android:layout_height=\"wrap_content\"\n"
                                        + "    android:visibility=\"gone\" >\n"
                                        + "\n"
                                        + "    <RatingBar\n"
                                        + "        android:id=\"@+id/review_form_rating\"\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:layout_alignParentRight=\"true\"\n"
                                        + "        android:numStars=\"5\"\n"
                                        + "        android:stepSize=\"1\" />\n"
                                        + "\n"
                                        + "    <EditText\n"
                                        + "        android:id=\"@+id/review_form_title\"\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:layout_alignParentLeft=\"true\"\n"
                                        + "        android:layout_toLeftOf=\"@id/review_form_rating\"\n"
                                        + "        android:hint=\"@string/review_form_title_hint\"\n"
                                        + "        android:inputType=\"text|textCapSentences|textAutoCorrect\"\n"
                                        + "        android:lines=\"1\" />\n"
                                        + "\n"
                                        + "    <EditText\n"
                                        + "        android:id=\"@+id/review_form_review\"\n"
                                        + "        android:layout_width=\"match_parent\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:layout_below=\"@id/review_form_title\"\n"
                                        + "        android:gravity=\"top\"\n"
                                        + "        android:hint=\"@string/review_form_content_hint\"\n"
                                        + "        android:inputType=\"text|textCapSentences|textAutoCorrect|textMultiLine\"\n"
                                        + "        android:minLines=\"3\"\n"
                                        + "        android:singleLine=\"false\"\n"
                                        + "        android:visibility=\"gone\" />\n"
                                        + "\n"
                                        + "    <Button\n"
                                        + "        android:id=\"@+id/review_form_submit\"\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:layout_alignParentRight=\"true\"\n"
                                        + "        android:layout_below=\"@id/review_form_review\"\n"
                                        + "        android:text=\"@string/submit\"\n"
                                        + "        android:visibility=\"gone\" />\n"
                                        + "\n"
                                        + "    <Button\n"
                                        + "        android:id=\"@+id/review_form_cancel\"\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:layout_below=\"@id/review_form_review\"\n"
                                        + "        android:layout_toLeftOf=\"@id/review_form_submit\"\n"
                                        + "        android:text=\"@string/cancel\"\n"
                                        + "        android:visibility=\"gone\" />\n"
                                        + "\n"
                                        + "</RelativeLayout>\n"),
                        mButtonbar_values)
                .issues(ButtonDetector.ORDER)
                .run()
                .expectClean();
    }

    public void testCase() {
        lint().files(mButtonbar, mButtonbar_values)
                .issues(ButtonDetector.CASE)
                .run()
                .expect(
                        ""
                                + "res/values/buttonbar-values.xml:9: Warning: The standard Android way to capitalize Ok is \"OK\" (tip: use @android:string/ok instead) [ButtonCase]\n"
                                + "    <string name=\"resume2\"> Ok </string>\n"
                                + "                            ~~\n"
                                + "res/values/buttonbar-values.xml:10: Warning: The standard Android way to capitalize CANCEL is \"Cancel\" (tip: use @android:string/cancel instead) [ButtonCase]\n"
                                + "    <string name=\"giveup2\">\"CANCEL\"</string>\n"
                                + "                           ~~~~~~~~\n"
                                + "0 errors, 2 warnings\n");
    }

    public void testBack() {
        lint().files(mButtonbar, mButtonbar_values)
                .issues(ButtonDetector.BACK_BUTTON)
                .run()
                .expect(
                        ""
                                + "res/layout/buttonbar.xml:183: Warning: Back buttons are not standard on Android; see design guide's navigation section [BackButton]\n"
                                + "    <Button\n"
                                + "     ~~~~~~\n"
                                + "0 errors, 1 warnings\n");
    }

    public void testEnglishLocales() {
        lint().files(manifest().minSdk(14), mButtonbar4, mButtonbar_values2)
                .issues(ButtonDetector.ORDER)
                .run()
                .expect(
                        ""
                                + "res/layout-en-rGB/buttonbar.xml:12: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-en-rGB/buttonbar.xml:44: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-en-rGB/buttonbar.xml:92: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-en-rGB/buttonbar.xml:124: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-en-rGB/buttonbar.xml:140: Warning: OK button should be on the right (was \"Ok | CANCEL\", should be \"CANCEL | Ok\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-en-rGB/buttonbar.xml:156: Warning: OK button should be on the right (was \"OK | Abort\", should be \"Abort | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-en-rGB/buttonbar.xml:177: Warning: Cancel button should be on the left (was \"Send | Cancel\", should be \"Cancel | Send\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 7 warnings\n");
    }

    public void testOtherLocales() {
        lint().files(manifest().minSdk(14), mButtonbar5, mButtonbar_values3)
                .issues(ButtonDetector.ORDER)
                .run()
                .expect(
                        ""
                                + "res/layout-de/buttonbar.xml:12: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout-de/buttonbar.xml:44: Warning: OK button should be on the right (was \"OK | Cancel\", should be \"Cancel | OK\") [ButtonOrder]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 2 warnings\n");
    }

    public void testOtherLocales2() {
        lint().files(mButtonbar5, mButtonbar_values3)
                .issues(ButtonDetector.CASE)
                .run()
                .expectClean();
    }

    public void testButtonStyle() {
        lint().files(
                        manifest().minSdk(14),
                        mButtonbar,
                        mButtonbar2_class,
                        mButtonbar3_class,
                        mButtonbar_values)
                .issues(ButtonDetector.STYLE)
                .run()
                .expect(
                        ""
                                + "res/layout/buttonbar.xml:12: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:17: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:28: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:33: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:44: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:49: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:60: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:65: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:76: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:81: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:92: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:97: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:108: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:113: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:124: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:129: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:140: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:145: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:156: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:161: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 20 warnings\n");
    }

    public void testButtonStyleFix() {
        lint().files(
                        manifest().minSdk(14),
                        xml(
                                "res/layout/buttonbar.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<merge xmlns:android=\"http://schemas.android.com/apk/res/android\">\n"
                                        + "\n"
                                        + "    <!-- Hardcoded strings, wrong order -->\n"
                                        + "    <LinearLayout\n"
                                        + "        android:layout_width=\"match_parent\"\n"
                                        + "        android:layout_height=\"wrap_content\" >\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"OK\" />\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"Cancel\" />\n"
                                        + "    </LinearLayout>\n"
                                        + "</merge>\n"))
                .issues(ButtonDetector.STYLE)
                .run()
                .expect(
                        ""
                                + "res/layout/buttonbar.xml:9: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "res/layout/buttonbar.xml:14: Warning: Buttons in button bars should be borderless; use style=\"?android:attr/buttonBarButtonStyle\" (and ?android:attr/buttonBarStyle on the parent) [ButtonStyle]\n"
                                + "        <Button\n"
                                + "         ~~~~~~\n"
                                + "0 errors, 2 warnings")
                .expectFixDiffs(
                        ""
                                + "Fix for res/layout/buttonbar.xml line 9: Make borderless:\n"
                                + "@@ -7 +7\n"
                                + "+         style=\"?android:attr/buttonBarStyle\"\n"
                                + "@@ -11 +12\n"
                                + "+             style=\"?android:attr/buttonBarButtonStyle\"\n"
                                + "@@ -16 +18\n"
                                + "+             style=\"?android:attr/buttonBarButtonStyle\"\n"
                                + "Fix for res/layout/buttonbar.xml line 14: Make borderless:\n"
                                + "@@ -7 +7\n"
                                + "+         style=\"?android:attr/buttonBarStyle\"\n"
                                + "@@ -11 +12\n"
                                + "+             style=\"?android:attr/buttonBarButtonStyle\"\n"
                                + "@@ -16 +18\n"
                                + "+             style=\"?android:attr/buttonBarButtonStyle\"");
    }

    public void testYesNo() {
        String expected =
                ""
                        + "res/layout/yesno.xml:10: Warning: @android:string/yes actually returns \"OK\", not \"Yes\"; use @android:string/ok instead or create a local string resource for Yes [ButtonCase]\n"
                        + "        android:text=\"@android:string/yes\" />\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "res/layout/yesno.xml:15: Warning: @android:string/no actually returns \"Cancel\", not \"No\"; use @android:string/cancel instead or create a local string resource for No [ButtonCase]\n"
                        + "        android:text=\"@android:string/no\" />\n"
                        + "        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
                        + "0 errors, 2 warnings\n";
        lint().files(
                        manifest().minSdk(4),
                        xml(
                                "res/layout/yesno.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    android:layout_width=\"match_parent\"\n"
                                        + "    android:layout_height=\"match_parent\"\n"
                                        + "    android:orientation=\"vertical\" >\n"
                                        + "\n"
                                        + "    <Button\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:text=\"@android:string/yes\" />\n"
                                        + "\n"
                                        + "    <Button\n"
                                        + "        android:layout_width=\"wrap_content\"\n"
                                        + "        android:layout_height=\"wrap_content\"\n"
                                        + "        android:text=\"@android:string/no\" />\n"
                                        + "\n"
                                        + "</LinearLayout>\n"))
                .issues(ButtonDetector.CASE)
                .run()
                .expect(expected);
    }

    public void testIssue180417() {
        // Regression test for
        //   https://code.google.com/p/android/issues/detail?id=180417
        String expected =
                ""
                        + "res/values/buttonbar-values.xml:4: Warning: The standard Android way to capitalize Ok is \"OK\" (tip: use @android:string/ok instead) [ButtonCase]\n"
                        + "    <string name=\"ok\">\"Ok\"</string>\n"
                        + "                      ~~~~\n"
                        + "0 errors, 1 warnings\n";
        lint().files(
                        xml(
                                "res/values/buttonbar-values.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<resources>\n"
                                        + "\n"
                                        + "    <string name=\"ok\">\"Ok\"</string>\n"
                                        + "\n"
                                        + "</resources>\n"),
                        xml(
                                "res/layout/buttonbar.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                                        + "    android:layout_width=\"match_parent\"\n"
                                        + "    android:layout_height=\"match_parent\"\n"
                                        + "    android:orientation=\"vertical\" >\n"
                                        + "    <LinearLayout\n"
                                        + "        android:layout_width=\"match_parent\"\n"
                                        + "        android:layout_height=\"wrap_content\" >\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"@string/cancel\" />\n"
                                        + "\n"
                                        + "        <Button\n"
                                        + "            android:layout_width=\"wrap_content\"\n"
                                        + "            android:layout_height=\"wrap_content\"\n"
                                        + "            android:text=\"@string/ok\" />\n"
                                        + "    </LinearLayout>\n"
                                        + "</LinearLayout>\n"))
                .issues(ButtonDetector.CASE)
                .run()
                .expect(expected);
    }

    public void testIssue101279() {
        // Regression test for https://code.google.com/p/android/issues/detail?id=101279
        lint().files(
                        manifest().minSdk(4),
                        xml(
                                "res/layout/buttonbar5.xml",
                                ""
                                        + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                                        + "<RelativeLayout\n"
                                        + "\t\txmlns:android=\"http://schemas.android.com/apk/res/android\">\n"
                                        + "\n"
                                        + "\t<LinearLayout />\n"
                                        + "\n"
                                        + "\t<Button\n"
                                        + "\t\t\tandroid:layout_toRightOf=\"@+id/anyid\"\n"
                                        + "\t\t\tandroid:text=\"Cancel\"/>\n"
                                        + "\n"
                                        + "</RelativeLayout>\n"))
                .issues(ButtonDetector.STYLE)
                .run()
                .expectClean();
    }

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar =
            xml(
                    "res/layout/buttonbar.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    android:layout_width=\"match_parent\"\n"
                            + "    android:layout_height=\"match_parent\"\n"
                            + "    android:orientation=\"vertical\" >\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources with varying case, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume2\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup2\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/abort\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/send\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <Button\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:text=\"@string/goback\" />\n"
                            + "\n"
                            + "</LinearLayout>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar2 =
            xml(
                    "res/layout-v14/buttonbar.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    android:layout_width=\"match_parent\"\n"
                            + "    android:layout_height=\"match_parent\"\n"
                            + "    android:orientation=\"vertical\" >\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources with varying case, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume2\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup2\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/abort\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/send\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <Button\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:text=\"@string/goback\" />\n"
                            + "\n"
                            + "</LinearLayout>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar2_class =
            xml(
                    "res/layout/buttonbar2.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<RelativeLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    android:layout_width=\"wrap_content\"\n"
                            + "    android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "    <ProgressBar\n"
                            + "        android:id=\"@+id/loading_progress\"\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:layout_alignParentLeft=\"true\"\n"
                            + "        android:layout_alignParentTop=\"true\"\n"
                            + "        android:layout_marginBottom=\"60dip\"\n"
                            + "        android:layout_marginLeft=\"40dip\"\n"
                            + "        android:layout_marginTop=\"40dip\"\n"
                            + "        android:max=\"10000\" />\n"
                            + "\n"
                            + "    <TextView\n"
                            + "        android:id=\"@+id/text\"\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:layout_alignParentTop=\"true\"\n"
                            + "        android:layout_alignWithParentIfMissing=\"true\"\n"
                            + "        android:layout_marginBottom=\"60dip\"\n"
                            + "        android:layout_marginLeft=\"40dip\"\n"
                            + "        android:layout_marginTop=\"40dip\"\n"
                            + "        android:layout_toRightOf=\"@id/loading_progress\"\n"
                            + "        android:ellipsize=\"end\"\n"
                            + "        android:maxLines=\"3\"\n"
                            + "        android:paddingRight=\"120dip\"\n"
                            + "        android:text=\"@string/creating_instant_mix\"\n"
                            + "        android:textAppearance=\"?android:attr/textAppearanceMedium\" />\n"
                            + "\n"
                            + "    <Button\n"
                            + "        android:id=\"@+id/cancel\"\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:layout_alignParentLeft=\"true\"\n"
                            + "        android:layout_alignRight=\"@id/text\"\n"
                            + "        android:layout_below=\"@id/text\"\n"
                            + "        android:background=\"@null\"\n"
                            + "        android:text=\"@string/cancel\" />\n"
                            + "\n"
                            + "    <ImageView\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:layout_above=\"@id/cancel\"\n"
                            + "        android:layout_alignLeft=\"@id/cancel\"\n"
                            + "        android:layout_alignRight=\"@id/cancel\"\n"
                            + "        android:scaleType=\"fitXY\"\n"
                            + "        android:src=\"@drawable/menu_list_divider\" />\n"
                            + "\n"
                            + "</RelativeLayout>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar3 =
            xml(
                    "res/layout-v16/buttonbar.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    android:layout_width=\"match_parent\"\n"
                            + "    android:layout_height=\"match_parent\"\n"
                            + "    android:orientation=\"vertical\" >\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources with varying case, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume2\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup2\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/abort\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/send\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <Button\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:text=\"@string/goback\" />\n"
                            + "\n"
                            + "</LinearLayout>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar3_class =
            xml(
                    "res/layout/buttonbar3.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    android:layout_width=\"match_parent\"\n"
                            + "    android:layout_height=\"wrap_content\"\n"
                            + "    android:orientation=\"vertical\" >\n"
                            + "\n"
                            + "    <TextView\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:layout_gravity=\"center_horizontal\"\n"
                            + "        android:text=\"@string/weekpicker_title\"\n"
                            + "        android:textAppearance=\"?android:attr/textAppearanceMedium\" />\n"
                            + "\n"
                            + "    <RelativeLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:background=\"#ffffff\"\n"
                            + "        android:padding=\"6dip\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:id=\"@+id/set\"\n"
                            + "            android:layout_width=\"120dip\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:layout_alignParentLeft=\"true\"\n"
                            + "            android:text=\"@string/weekpicker_set\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:id=\"@+id/cancel\"\n"
                            + "            android:layout_width=\"120dip\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:layout_alignParentRight=\"true\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </RelativeLayout>\n"
                            + "\n"
                            + "</LinearLayout>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar4 =
            xml(
                    "res/layout-en-rGB/buttonbar.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    android:layout_width=\"match_parent\"\n"
                            + "    android:layout_height=\"match_parent\"\n"
                            + "    android:orientation=\"vertical\" >\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources with varying case, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume2\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup2\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/abort\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/send\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <Button\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:text=\"@string/goback\" />\n"
                            + "\n"
                            + "</LinearLayout>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar5 =
            xml(
                    "res/layout-de/buttonbar.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<LinearLayout xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    android:layout_width=\"match_parent\"\n"
                            + "    android:layout_height=\"match_parent\"\n"
                            + "    android:orientation=\"vertical\" >\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Hardcoded strings, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"Cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"OK\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @android:string resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@android:string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- @string/ok/cancel resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, right order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Random name resources with varying case, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/resume2\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/giveup2\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/ok\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/abort\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <!-- Resources with only one of OK and Cancel, wrong order -->\n"
                            + "    <LinearLayout\n"
                            + "        android:layout_width=\"match_parent\"\n"
                            + "        android:layout_height=\"wrap_content\" >\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/send\" />\n"
                            + "\n"
                            + "        <Button\n"
                            + "            android:layout_width=\"wrap_content\" android:background=\"?android:attr/selectableItemBackground\"\n"
                            + "            android:layout_height=\"wrap_content\"\n"
                            + "            android:text=\"@string/cancel\" />\n"
                            + "    </LinearLayout>\n"
                            + "\n"
                            + "    <Button\n"
                            + "        android:layout_width=\"wrap_content\"\n"
                            + "        android:layout_height=\"wrap_content\"\n"
                            + "        android:text=\"@string/goback\" />\n"
                            + "\n"
                            + "</LinearLayout>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar_values =
            xml(
                    "res/values/buttonbar-values.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<resources>\n"
                            + "\n"
                            + "    <string name=\"button\"> Button </string>\n"
                            + "    <string name=\"ok\"> OK </string>\n"
                            + "    <string name=\"cancel\"> Cancel </string>\n"
                            + "    <string name=\"resume\"> OK </string>\n"
                            + "    <string name=\"giveup\"> Cancel </string>\n"
                            + "    <string name=\"resume2\"> Ok </string>\n"
                            + "    <string name=\"giveup2\">\"CANCEL\"</string>\n"
                            + "    <string name=\"send\"> Send </string>\n"
                            + "    <string name=\"abort\">Abort</string>\n"
                            + "    <string name=\"goback\">'Back'</string>\n"
                            + "\n"
                            + "</resources>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar_values2 =
            xml(
                    "res/values-en-rGB/buttonbar-values.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<resources>\n"
                            + "\n"
                            + "    <string name=\"button\"> Button </string>\n"
                            + "    <string name=\"ok\"> OK </string>\n"
                            + "    <string name=\"cancel\"> Cancel </string>\n"
                            + "    <string name=\"resume\"> OK </string>\n"
                            + "    <string name=\"giveup\"> Cancel </string>\n"
                            + "    <string name=\"resume2\"><![CDATA[ Ok ]]></string>\n"
                            + "    <string name=\"giveup2\">\"CANCEL\"</string>\n"
                            + "    <string name=\"send\"> Send </string>\n"
                            + "    <string name=\"abort\">Abort</string>\n"
                            + "    <string name=\"goback\">'Back'</string>\n"
                            + "\n"
                            + "</resources>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mButtonbar_values3 =
            xml(
                    "res/values-de/buttonbar-values.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<resources>\n"
                            + "\n"
                            + "    <string name=\"button\"> Button </string>\n"
                            + "    <string name=\"ok\"> OK </string>\n"
                            + "    <string name=\"cancel\"> Cancel </string>\n"
                            + "    <string name=\"resume\"> OK </string>\n"
                            + "    <string name=\"giveup\"> Cancel </string>\n"
                            + "    <string name=\"resume2\"> Ok </string>\n"
                            + "    <string name=\"giveup2\">\"CANCEL\"</string>\n"
                            + "    <string name=\"send\"> Send </string>\n"
                            + "    <string name=\"abort\">Abort</string>\n"
                            + "    <string name=\"goback\">'Back'</string>\n"
                            + "\n"
                            + "</resources>\n");

    @SuppressWarnings("all") // Sample code
    private TestFile mMinsdk5targetsdk14 =
            xml(
                    "AndroidManifest.xml",
                    ""
                            + "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                            + "<manifest xmlns:android=\"http://schemas.android.com/apk/res/android\"\n"
                            + "    package=\"test.bytecode\"\n"
                            + "    android:versionCode=\"1\"\n"
                            + "    android:versionName=\"1.0\" >\n"
                            + "\n"
                            + "    <uses-sdk android:minSdkVersion=\"5\" android:targetSdkVersion=\"14\" />\n"
                            + "\n"
                            + "    <application\n"
                            + "        android:icon=\"@drawable/ic_launcher\"\n"
                            + "        android:label=\"@string/app_name\" >\n"
                            + "        <activity\n"
                            + "            android:name=\".BytecodeTestsActivity\"\n"
                            + "            android:label=\"@string/app_name\" >\n"
                            + "            <intent-filter>\n"
                            + "                <action android:name=\"android.intent.action.MAIN\" />\n"
                            + "\n"
                            + "                <category android:name=\"android.intent.category.LAUNCHER\" />\n"
                            + "            </intent-filter>\n"
                            + "        </activity>\n"
                            + "    </application>\n"
                            + "\n"
                            + "</manifest>\n");
}
