/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.lint

import com.android.tools.lint.Reporter.Companion.encodeUrl
import com.android.tools.lint.checks.BuiltinIssueRegistry
import com.android.tools.lint.client.api.LintClient
import com.android.tools.lint.detector.api.Category
import com.android.tools.lint.detector.api.Issue
import java.lang.reflect.Field
import junit.framework.TestCase

class ReporterTest : TestCase() {
  override fun setUp() {
    LintClient.clientName = LintClient.CLIENT_UNIT_TESTS
  }

  fun testEncodeUrl() {
    assertEquals("a/b/c", encodeUrl("a/b/c"))
    assertEquals("a/b/c", encodeUrl("a\\b\\c"))
    assertEquals(
      "a/b/c/%24%26%2B%2C%3A%3B%3D%3F%40/foo+bar%25/d",
      encodeUrl("a/b/c/$&+,:;=?@/foo bar%/d"),
    )
    assertEquals("a/%28b%29/d", encodeUrl("a/(b)/d"))
    assertEquals("a/b+c/d", encodeUrl("a/b c/d")) // + or %20
  }

  fun testHasQuickFix() {
    // This test is used to generate a list of detector+field names for the Reporter#hasAutoFix
    // database.

    // List of issue id's where the unit test suite encountered a quickfix. This list can
    // be generated by applying a diff like this to TestLintClient, then running the test suite
    // and then picking up the unique (cat /tmp/fix.txt | sort | uniq) id's and pasting them
    // into the below string literal:
    /*
       diff --git a/lint/libs/lint-tests/src/main/java/com/android/tools/lint/checks/infrastructure/TestLintClient.java b/lint/libs/lint-tests/src/main/java/com/android/tools/lint/checks/infrastructure/TestLintClient.java
       index e21784a802..58b0c261b5 100644
       --- a/lint/libs/lint-tests/src/main/java/com/android/tools/lint/checks/infrastructure/TestLintClient.java
       +++ b/lint/libs/lint-tests/src/main/java/com/android/tools/lint/checks/infrastructure/TestLintClient.java
       @@ -989,6 +989,14 @@ public class TestLintClient extends LintCliClient {
                Issue issue = incident.getIssue();
                assertNotNull(location);

       +        if (fix != null && !(fix instanceof LintFix.ShowUrl)) {
       +            File file = new File("/tmp/fix.txt");
       +            FilesKt.appendText(file, incident.getIssue().getId(), kotlin.text.Charsets.UTF_8);
       +        }
       +
                // Ensure that we're inside a read action if we might need access to PSI.
                // This is one heuristic; we could add more assertions elsewhere as needed.
                if (context instanceof JavaContext
    */
    // Then run the below code with the list of issue id's to search for detector names and field
    // references
    val ids = ""
    val registry = BuiltinIssueRegistry()
    val fieldMap = mutableMapOf<String, Field>()
    for (id in ids.split("\n")) {
      val issue = registry.getIssue(id)
      if (issue == null) {
        println("New issue found in the registry for $id")
        continue
      }
      val field = findField(issue)
      if (field != null) {
        fieldMap[id] = field
      }
    }

    for (id in ids.split("\n")) {
      val issue = registry.getIssue(id) ?: continue
      if (issue.category == Category.LINT) {
        continue
      }
      val field = fieldMap[id] ?: continue
      println(issue.implementation.detectorClass.simpleName + "." + field.name + ",")
    }
  }

  private fun findField(issue: Issue): Field? {
    val id = issue.id
    val clz: Class<*> = issue.implementation.detectorClass
    findField(id, clz)?.let {
      return it
    }

    try {
      val companion: Class<*> = Class.forName(clz.name + ".Companion")
      findField(id, companion)?.let {
        return it
      }
    } catch (ignore: Throwable) {}

    try {
      val companion: Class<*> = Class.forName(clz.name + ".Kt")
      findField(id, companion)?.let {
        return it
      }
    } catch (ignore: Throwable) {}

    return null
  }

  private fun findField(id: String, clz: Class<*>): Field? {
    val fields = clz.declaredFields
    for (field in fields) {
      try {
        if (field.type == Issue::class.java) {
          field.isAccessible = true
          val issue = field.get(null) as? Issue
          if (issue?.id == id) {
            return field
          }
        }
      } catch (ignore: Throwable) {}
    }

    return null
  }
}
