/*
 * Copyright (C) 2010 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.resources;

import junit.framework.TestCase;

public class FolderTypeRelationShipTest extends TestCase {

    public void testResourceType() {
        // All resource type except AAPT and SAMPLE_DATA should be in the FolderTypeRelationShip map.
        // Loop on all of the enum, and make sure there's at least one folder type for it.
        for (ResourceType type : ResourceType.values()) {
            if (type == ResourceType.AAPT
                    || type == ResourceType.SAMPLE_DATA
                    || type == ResourceType.OVERLAYABLE) {
                continue;
            }
            assertTrue(type.getDisplayName(),
                    !FolderTypeRelationship.getRelatedFolders(type).isEmpty());
        }
    }

    public void testResourceFolderType() {
        // All resource folder type should generate at least one type of resource.
        // Loop on all of the enum, and make sure there's at least one res type for it.
        for (ResourceFolderType type : ResourceFolderType.values()) {
            assertTrue(type.getName(),
                    !FolderTypeRelationship.getRelatedResourceTypes(type).isEmpty());
        }
    }

    public void testIsIdGeneratingFolderType() {
        // Sample a few.
        assertTrue(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.LAYOUT));
        assertTrue(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.MENU));
        assertTrue(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.DRAWABLE));
        assertTrue(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.XML));
        assertTrue(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.NAVIGATION));

        assertFalse(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.COLOR));
        assertFalse(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.MIPMAP));
        assertFalse(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.RAW));
        assertFalse(FolderTypeRelationship.isIdGeneratingFolderType(ResourceFolderType.VALUES));
    }

    public void testGetNonIdRelatedResourceType() {
        // Check that each folder type can generate a non-ID resource type and afterwards verify
        // that the produced resource type can be indeed generated by this folder type.
        for (ResourceFolderType folderType : ResourceFolderType.values()) {
            ResourceType resourceType =
                    FolderTypeRelationship.getNonIdRelatedResourceType(folderType);
            assertFalse(resourceType == ResourceType.ID);
            assertTrue(FolderTypeRelationship.getRelatedFolders(resourceType).contains(folderType));

            // The id shouldn't be the first thing we return
            assertNotSame(ResourceType.ID,
                          FolderTypeRelationship.getRelatedResourceTypes(folderType).iterator().next());
        }

        // Sample a few.
        assertEquals(
                ResourceType.DRAWABLE,
                FolderTypeRelationship.getNonIdRelatedResourceType(ResourceFolderType.DRAWABLE));
        assertEquals(
                ResourceType.MENU,
                FolderTypeRelationship.getNonIdRelatedResourceType(ResourceFolderType.MENU));
        assertEquals(
                ResourceType.RAW,
                FolderTypeRelationship.getNonIdRelatedResourceType(ResourceFolderType.RAW));
    }
}
