/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.flags.junit;

import com.android.flags.Flag;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.junit.rules.ExternalResource;

/**
 * Sets a flag to the given value, if provided, before running a test and restores it to
 * the original value after the test.
 *
 * <pre>
 *   public class MyTest {
 *    {@literal @}Rule
 *     public TestRule myFlagRule = new FlagRule&lt;&gt;(StudioFlags.MY_FLAG, true);
 *   }
 * </pre>
 */
public class FlagRule<T> extends ExternalResource {
    private final @NotNull Flag<T> myFlag;
    private final @Nullable T myValue;

    public FlagRule(@NotNull Flag<T> flag, @NotNull T value) {
        myFlag = flag;
        myValue = value;
    }

    public FlagRule(@NotNull Flag<T> flag) {
        myFlag = flag;
        myValue = null;
    }

    @Override
    protected void before() {
        if (myValue != null) {
            myFlag.override(myValue);
        }
    }

    @Override
    protected void after() {
        myFlag.clearOverride();
    }
}
