/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.internal.tasks

import com.android.build.api.variant.impl.TestVariantImpl
import com.android.build.gradle.internal.AvdComponentsBuildService
import com.android.build.gradle.internal.ManagedVirtualDeviceLockManager
import com.android.build.gradle.internal.SdkComponentsBuildService
import com.android.build.gradle.internal.SdkComponentsBuildService.VersionedSdkLoader
import com.android.build.gradle.internal.component.DeviceTestCreationConfig
import com.android.build.gradle.internal.dsl.EmulatorControl
import com.android.build.gradle.internal.dsl.EmulatorSnapshots
import com.android.build.gradle.internal.dsl.ManagedVirtualDevice
import com.android.build.gradle.internal.fixtures.FakeGradleProperty
import com.android.build.gradle.internal.fixtures.FakeGradleProvider
import com.android.build.gradle.internal.fixtures.FakeGradleWorkExecutor
import com.android.build.gradle.internal.profile.AnalyticsService
import com.android.build.gradle.internal.tasks.factory.GlobalTaskCreationConfigImpl
import com.android.build.gradle.internal.test.AbstractTestDataImpl
import com.android.build.gradle.internal.testing.utp.EmulatorControlConfig
import com.android.build.gradle.internal.testing.utp.ManagedDeviceTestRunner
import com.android.build.gradle.internal.testing.utp.RetentionConfig
import com.android.build.gradle.internal.testing.utp.UtpDependencies
import com.android.build.gradle.options.BooleanOption
import com.android.builder.model.TestOptions
import com.android.repository.Revision
import com.android.testutils.MockitoKt
import com.android.testutils.MockitoKt.any
import com.android.testutils.MockitoKt.argThat
import com.android.testutils.MockitoKt.eq
import com.google.common.truth.Truth.assertThat
import org.gradle.api.GradleException
import org.gradle.api.Project
import org.gradle.api.artifacts.ArtifactCollection
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.Directory
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.RegularFile
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.internal.TaskOutputsInternal
import org.gradle.api.logging.Logger
import org.gradle.api.provider.ListProperty
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.gradle.api.services.BuildServiceRegistration
import org.gradle.testfixtures.ProjectBuilder
import org.gradle.workers.WorkerExecutor
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.rules.TemporaryFolder
import org.mockito.Answers.CALLS_REAL_METHODS
import org.mockito.Answers.RETURNS_DEEP_STUBS
import org.mockito.Mock
import org.mockito.Mockito.doReturn
import org.mockito.Mockito.mock
import org.mockito.Mockito.verify
import org.mockito.Mockito.verifyNoInteractions
import org.mockito.Mockito.verifyNoMoreInteractions
import org.mockito.Mockito.`when`
import org.mockito.junit.MockitoJUnit
import java.io.File
import java.util.logging.Level

class ManagedDeviceInstrumentationTestTaskTest {
    private lateinit var mockVersionedSdkLoader: VersionedSdkLoader

    private lateinit var utpJvm: File
    private lateinit var emulatorFile: File
    private lateinit var avdFolder: File
    private lateinit var resultsFolder: File
    private lateinit var codeCoverage: File
    private lateinit var reportsFolder: File

    @Mock
    lateinit var emulatorDirectory: Directory
    @Mock
    lateinit var avdDirectory: Directory
    @Mock
    lateinit var resultsDirectory: Directory
    @Mock
    lateinit var coverageDirectory: Directory
    @Mock
    lateinit var reportsDirectory: Directory
    @Mock
    lateinit var utpJvmFile: RegularFile

    @get:Rule
    val mockitoRule = MockitoJUnit.rule()

    @get:Rule
    val temporaryFolderRule = TemporaryFolder()

    @Mock
    lateinit var avdService: AvdComponentsBuildService

    @Mock
    lateinit var sdkService: SdkComponentsBuildService

    @Mock(answer = RETURNS_DEEP_STUBS, extraInterfaces = [DeviceTestCreationConfig::class])
    lateinit var creationConfig: TestVariantImpl

    @Mock(answer = RETURNS_DEEP_STUBS)
    lateinit var globalConfig: GlobalTaskCreationConfigImpl

    @Mock(answer = RETURNS_DEEP_STUBS)
    lateinit var testData: AbstractTestDataImpl

    @Mock
    lateinit var runnerFactory: ManagedDeviceInstrumentationTestTask.TestRunnerFactory

    @Mock
    lateinit var intallOptions: ListProperty<String>

    @Mock
    lateinit var dependencies: ArtifactCollection

    private lateinit var project: Project
    private lateinit var workerExecutor: WorkerExecutor

    @Before
    fun setup() {
        `when`(creationConfig.computeTaskNameInternal(any(), any())).then {
            val prefix = it.getArgument<String>(0)
            val suffix = it.getArgument<String>(0)
            "${prefix}AndroidDebugTest$suffix"
        }
        `when`(creationConfig.name).thenReturn("AndroidDebugTest")

        // Setup Build Services for configuration.
        val mockGeneralRegistration = mock(BuildServiceRegistration::class.java, RETURNS_DEEP_STUBS)
        `when`(creationConfig.services.buildServiceRegistry.registrations.getByName(any()))
            .thenReturn(mockGeneralRegistration)

        mockVersionedSdkLoader = mock(VersionedSdkLoader::class.java)
        `when`(mockVersionedSdkLoader.offlineMode).thenReturn(false)

        `when`(sdkService.sdkLoader(any(), any())).thenReturn(mockVersionedSdkLoader)

        emulatorFile = temporaryFolderRule.newFolder("emulator")
        `when`(emulatorDirectory.asFile).thenReturn(emulatorFile)
        `when`(avdService.emulatorDirectory).thenReturn(FakeGradleProvider(emulatorDirectory))

        utpJvm = temporaryFolderRule.newFile("java")

        avdFolder = temporaryFolderRule.newFolder("gradle/avd")
        `when`(avdDirectory.asFile).thenReturn(avdFolder)
        `when`(avdService.avdFolder).thenReturn(FakeGradleProvider(avdDirectory))

        val lockManager = mock(ManagedVirtualDeviceLockManager::class.java)
        val lock = mock(ManagedVirtualDeviceLockManager.DeviceLock::class.java)
        `when`(lock.lockCount).thenReturn(1)
        `when`(lockManager.lock(any())).thenReturn(lock)
        `when`(avdService.lockManager).thenReturn(lockManager)

        reportsFolder = temporaryFolderRule.newFolder("reports")
        `when`(reportsDirectory.asFile).thenReturn(reportsFolder)

        resultsFolder = temporaryFolderRule.newFolder("results")
        `when`(resultsDirectory.asFile).thenReturn(resultsFolder)

        codeCoverage = temporaryFolderRule.newFolder("coverage")
        `when`(coverageDirectory.asFile).thenReturn(codeCoverage)

        project = ProjectBuilder.builder().withProjectDir(temporaryFolderRule.newFolder()).build()
        workerExecutor = FakeGradleWorkExecutor(project.objects, temporaryFolderRule.newFolder())
    }

    private fun <T> mockEmptyProperty(): Property<T> {
        @Suppress("UNCHECKED_CAST")
        return mock(Property::class.java) as Property<T>
    }

    private fun mockDirectoryProperty(directory: Directory): DirectoryProperty {
        val property = mock(DirectoryProperty::class.java)
        `when`(property.get()).thenReturn(directory)
        return property
    }

    private fun mockFileProperty(file: RegularFile): RegularFileProperty =
        mock(RegularFileProperty::class.java).apply {
            `when`(get()).thenReturn(file)
        }

    private fun basicTaskSetup(): ManagedDeviceInstrumentationTestTask {

        val task = mock(
            ManagedDeviceInstrumentationTestTask::class.java,
            CALLS_REAL_METHODS)

        doReturn(FakeGradleProperty(mock(AnalyticsService::class.java)))
            .`when`(task).analyticsService
        doReturn(FakeGradleProperty("project_path")).`when`(task).projectPath

        doReturn("path").`when`(task).path
        doReturn(mock(TaskOutputsInternal::class.java, RETURNS_DEEP_STUBS))
            .`when`(task).outputs
        doReturn(MockitoKt.mock<Logger>()).`when`(task).logger

        doReturn(runnerFactory).`when`(task).testRunnerFactory
        doReturn(FakeGradleProperty(avdService)).`when`(runnerFactory).avdComponents
        doReturn(FakeGradleProperty(testData)).`when`(task).testData
        doReturn(mock(ListProperty::class.java)).`when`(task).installOptions
        val mockManagedDevice = mock(ManagedVirtualDevice::class.java)
        doReturn("testDevice1").`when`(mockManagedDevice).getName()
        doReturn(29).`when`(mockManagedDevice).apiLevel
        doReturn("aosp").`when`(mockManagedDevice).systemImageSource
        doReturn(false).`when`(mockManagedDevice).require64Bit
        doReturn(FakeGradleProperty(mockManagedDevice)).`when`(task).device
        doReturn(FakeGradleProperty(false)).`when`(task).enableEmulatorDisplay
        doReturn(FakeGradleProperty(false)).`when`(task).getAdditionalTestOutputEnabled()
        doReturn(dependencies).`when`(task).dependencies

        doReturn(FakeGradleProperty(true))
            .`when`(testData).hasTests(any(), any(), any())
        doReturn(FakeGradleProperty("flavor_name"))
            .`when`(testData).flavorName

        val buddyApks = mock(ConfigurableFileCollection::class.java)
        `when`(buddyApks.files).thenReturn(setOf())
        `when`(task.buddyApks).thenReturn(buddyApks)

        `when`(task.installOptions).thenReturn(intallOptions)
        `when`(intallOptions.getOrElse(any())).thenReturn(listOf())

        doReturn(mockDirectoryProperty(resultsDirectory)).`when`(task).resultsDir
        doReturn(mockDirectoryProperty(coverageDirectory)).`when`(task).getCoverageDirectory()
        doReturn(mockDirectoryProperty(reportsDirectory)).`when`(task).getReportsDir()

        doReturn(workerExecutor).`when`(task).workerExecutor

        return task
    }

    @Test
    fun testRunnerFactory_testCreateTestRunner() {
        val factory = mock(
            ManagedDeviceInstrumentationTestTask.TestRunnerFactory::class.java,
            CALLS_REAL_METHODS)

        `when`(factory.executionEnum)
            .thenReturn(FakeGradleProperty(TestOptions.Execution.ANDROIDX_TEST_ORCHESTRATOR))
        `when`(factory.forceCompilation)
            .thenReturn(FakeGradleProperty(false))
        `when`(factory.retentionConfig)
            .thenReturn(FakeGradleProperty(mock(RetentionConfig::class.java)))
        `when`(factory.emulatorControlConfig)
            .thenReturn(FakeGradleProperty(mock(EmulatorControlConfig::class.java)))
        `when`(factory.compileSdkVersion).thenReturn(FakeGradleProperty("sdkVersion"))
        `when`(factory.buildToolsRevision)
            .thenReturn(FakeGradleProperty(mock(Revision::class.java)))
        `when`(factory.testShardsSize).thenReturn(FakeGradleProperty(null))
        `when`(factory.sdkBuildService).thenReturn(FakeGradleProperty(sdkService))
        `when`(factory.avdComponents).thenReturn(FakeGradleProperty(avdService))
        `when`(factory.utpDependencies).thenReturn(mock(UtpDependencies::class.java))
        `when`(factory.utpLoggingLevel)
            .thenReturn(FakeGradleProperty(Level.OFF))
        `when`(factory.emulatorGpuFlag).thenReturn(FakeGradleProperty("auto-no-window"))
        `when`(factory.showEmulatorKernelLoggingFlag).thenReturn(FakeGradleProperty(false))
        `when`(factory.installApkTimeout).thenReturn(FakeGradleProperty(0))
        `when`(factory.enableEmulatorDisplay).thenReturn(FakeGradleProperty(false))
        `when`(factory.getTargetIsSplitApk).thenReturn(FakeGradleProperty(false))
        `when`(factory.getKeepInstalledApks).thenReturn(FakeGradleProperty(false))
        doReturn(mockFileProperty(utpJvmFile)).`when`(factory).jvmExecutable
        doReturn(utpJvm).`when`(utpJvmFile).asFile

        val testRunner = factory.createTestRunner(workerExecutor, null)
        assertThat(testRunner).isInstanceOf(ManagedDeviceTestRunner::class.java)

        factory.createTestRunner(workerExecutor, null)
    }

    @Test
    fun creationConfig_testTaskConfiguration() {
        // Parameters for config class.
        val resultsDir = temporaryFolderRule.newFolder("resultsDir")
        val reportsDir = temporaryFolderRule.newFolder("reportsDir")
        val additionalTestOutputDir = temporaryFolderRule.newFolder("additionalTestOutputDir")
        val coverageOutputDir = temporaryFolderRule.newFolder("coverageOutputDir")
        val managedDevice = ManagedVirtualDevice("someNameHere").also {
            it.device = "Pixel 2"
            it.apiLevel = 27
            it.systemImageSource = "aosp"
        }
        val emulatorControl = mock(EmulatorControl::class.java)
        // Needed for cast from api class to internal class
        val snapshots = mock(EmulatorSnapshots::class.java)
        `when`(creationConfig.global.androidTestOptions.emulatorSnapshots).thenReturn(snapshots)
        `when`(creationConfig.global.androidTestOptions.emulatorControl).thenReturn(emulatorControl)
        // Needed to ensure that UTP is active
        `when`(
            creationConfig.services
                .projectOptions[BooleanOption.ANDROID_TEST_USES_UNIFIED_TEST_PLATFORM])
            .thenReturn(true)
        // Needed to ensure the ExecutionEnum
        `when`(creationConfig.global.testOptionExecutionEnum)
            .thenReturn(TestOptions.Execution.ANDROIDX_TEST_ORCHESTRATOR)
        `when`(creationConfig
                .services
                .projectOptions
                .get(BooleanOption.PRIVACY_SANDBOX_SDK_SUPPORT)).thenReturn(true)

        `when`(testData.privacySandboxSdkApks)
                .thenReturn(project.objects.fileCollection())
        val config = ManagedDeviceInstrumentationTestTask.CreationAction(
            creationConfig,
            managedDevice,
            testData,
            resultsDir,
            reportsDir,
            additionalTestOutputDir,
            coverageOutputDir,
            ""
        )

        val task =
            mock(ManagedDeviceInstrumentationTestTask::class.java, RETURNS_DEEP_STUBS)

        // We need to create mock properties to verify/capture values in the task as
        // RETURNS_DEEP_STUBS does not work as expected with verify. Also, we can't use
        // FakeGradleProperties because they do not support disallowChanges().

        val executionEnum = mockEmptyProperty<TestOptions.Execution>()
        `when`(task.testRunnerFactory.executionEnum).thenReturn(executionEnum)
        val sdkBuildService = mockEmptyProperty<SdkComponentsBuildService>()
        `when`(task.testRunnerFactory.sdkBuildService).thenReturn(sdkBuildService)
        val avdComponents = mockEmptyProperty<AvdComponentsBuildService>()
        `when`(task.testRunnerFactory.avdComponents).thenReturn(avdComponents)

        val device = mockEmptyProperty<ManagedVirtualDevice>()
        `when`(task.device).thenReturn(device)

        config.configure(task)

        verify(executionEnum).set(TestOptions.Execution.ANDROIDX_TEST_ORCHESTRATOR)
        verify(executionEnum).disallowChanges()
        verifyNoMoreInteractions(executionEnum)

        verify(sdkBuildService).set(any<Provider<SdkComponentsBuildService>>())
        verify(sdkBuildService).disallowChanges()
        verifyNoMoreInteractions(sdkBuildService)

        verify(device).set(managedDevice)
        verify(device).disallowChanges()
        verifyNoMoreInteractions(device)

        verify(avdComponents).set(any<Provider<AvdComponentsBuildService>>())
        verify(avdComponents).disallowChanges()
        verifyNoMoreInteractions(avdComponents)
    }

    @Test
    fun taskAction_basicTaskPath() {
        val task = basicTaskSetup()

        val testRunner = mock(ManagedDeviceTestRunner::class.java)
        doReturn(true).`when`(testRunner).runTests(
            managedDevice = any(),
            runId = any(),
            outputDirectory = any(),
            coverageOutputDirectory = any(),
            additionalTestOutputDir = eq(null),
            projectPath = any(),
            variantName = any(),
            testData = any(),
            additionalInstallOptions = any(),
            helperApks = any(),
            logger = any(),
            dependencyApks = any()
        )
        println("TestRunner: $testRunner")

        doReturn(FakeGradleProperty<Int>()).`when`(runnerFactory).testShardsSize
        doReturn(testRunner).`when`(runnerFactory).createTestRunner(any(), eq(null))
        `when`(runnerFactory.executionEnum)
            .thenReturn(FakeGradleProperty(TestOptions.Execution.ANDROIDX_TEST_ORCHESTRATOR))

        task.doTaskAction()

        verify(testRunner).runTests(
            managedDevice = argThat {
                it.getName() == "testDevice1"
                        && it is ManagedVirtualDevice
                        && it.apiLevel == 29
                        && it.systemImageSource == "aosp"
                        && it.require64Bit == false
            },
            runId = any(),
            outputDirectory = eq(resultsFolder),
            coverageOutputDirectory = eq(codeCoverage),
            additionalTestOutputDir = eq(null),
            projectPath = eq("project_path"),
            variantName = eq("flavor_name"),
            testData = any(),
            additionalInstallOptions = eq(listOf()),
            helperApks = any(),
            logger = any(),
            dependencyApks = any()
        )
        verifyNoMoreInteractions(testRunner)
    }

    @Test
    fun taskAction_testFailuresPath() {
        val task = basicTaskSetup()

        val testRunner = mock(ManagedDeviceTestRunner::class.java)
        doReturn(false).`when`(testRunner).runTests(
            managedDevice = any(),
            runId = any(),
            outputDirectory = any(),
            coverageOutputDirectory = any(),
            additionalTestOutputDir = eq(null),
            projectPath = any(),
            variantName = any(),
            testData = any(),
            additionalInstallOptions = any(),
            helperApks = any(),
            logger = any(),
            dependencyApks = any()
        )

        doReturn(FakeGradleProperty<Int>()).`when`(runnerFactory).testShardsSize

        doReturn(testRunner).`when`(runnerFactory).createTestRunner(any(), eq(null))
        `when`(runnerFactory.executionEnum)
            .thenReturn(FakeGradleProperty(TestOptions.Execution.ANDROIDX_TEST_ORCHESTRATOR))

        task.setIgnoreFailures(false)

        try {
            task.doTaskAction()

            error("Should not reach.")
        } catch (e: GradleException) {
            assertThat(e.message).startsWith("There were failing tests")
        }

        verify(testRunner).runTests(
            managedDevice = argThat {
                it.getName() == "testDevice1"
                        && it is ManagedVirtualDevice
                        && it.apiLevel == 29
                        && it.systemImageSource == "aosp"
                        && it.require64Bit == false
            },
            runId = any(),
            outputDirectory = eq(resultsFolder),
            coverageOutputDirectory = eq(codeCoverage),
            additionalTestOutputDir = eq(null),
            projectPath = eq("project_path"),
            variantName = eq("flavor_name"),
            testData = any(),
            additionalInstallOptions = eq(listOf()),
            helperApks = any(),
            logger = any(),
            dependencyApks = any()
        )
        verifyNoMoreInteractions(testRunner)
    }

    @Test
    fun taskAction_noTestsPath() {
        val task = basicTaskSetup()

        val testRunner = mock(ManagedDeviceTestRunner::class.java)
        doReturn(false).`when`(testRunner).runTests(
            managedDevice = any(),
            runId = any(),
            outputDirectory = any(),
            coverageOutputDirectory = any(),
            additionalTestOutputDir = eq(null),
            projectPath = any(),
            variantName = any(),
            testData = any(),
            additionalInstallOptions = any(),
            helperApks = any(),
            logger = any(),
            dependencyApks = any()
        )

        doReturn(FakeGradleProperty<Int>()).`when`(runnerFactory).testShardsSize

        doReturn(testRunner).`when`(runnerFactory).createTestRunner(any(), eq(null))
        `when`(runnerFactory.executionEnum)
            .thenReturn(FakeGradleProperty(TestOptions.Execution.ANDROIDX_TEST_ORCHESTRATOR))

        // When the data has no Tests, the testRunner should not be run.
        doReturn(FakeGradleProperty(false))
            .`when`(testData).hasTests(any(), any(), any())

        task.doTaskAction()

        verifyNoInteractions(testRunner)
    }
}
