/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.internal.pipeline;

import static org.junit.Assert.fail;

import com.android.annotations.NonNull;
import com.google.common.collect.Lists;
import java.util.List;
import java.util.Set;
import java.util.function.ToIntFunction;


public final class BitMaskTestUtils {

    private BitMaskTestUtils() {}

    /**
     * Validator for the correctness of a bit mask function.
     *
     * <p>Asserts that the bit masks of all of the items in the {@code itemsSet} as generated by the
     * {@code bitMaskFunction} are unique.
     */
    public static <T> void checkScopeBitMaskUnique(
            @NonNull Set<T> itemsSet,
            @NonNull ToIntFunction<T> bitMaskFunction) {
        List<T> items = Lists.newArrayList(itemsSet);
        for (int i = 0; i < items.size(); i++) {
            T item1 = items.get(i);
            int mask1 = bitMaskFunction.applyAsInt(item1);
            if (mask1 == 0) {
                fail(String.format("Bit mask for %1$s is zero.", item1));
            }
            for (int j = 0; j < i; j++) {
                T item2 = items.get(j);
                int mask2 = bitMaskFunction.applyAsInt(item2);
                if ((mask1 & mask2) != 0) {
                    fail(String.format("Bit masks for:\n"
                            + "  %1$s [%2$s] and \n"
                            + "  %3$s [%4$s] are not unique",
                            item1, Integer.toBinaryString(mask1),
                            item2, Integer.toBinaryString(mask2)));
                }
            }
        }
    }

}
