package org.jetbrains.compose.reload.gradle
import java.nio.file.Path
import kotlin.io.path.Path
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.jetbrains.compose.reload.core.HotReloadProperty
import org.jetbrains.compose.reload.core.Os
import org.jetbrains.compose.reload.InternalHotReloadApi
import org.jetbrains.compose.reload.DelicateHotReloadApi
import org.jetbrains.compose.reload.ExperimentalHotReloadApi


  /**
* See [HotReloadProperty.OrchestrationPort]
* Current 'Orchestration Server' port. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...)
*/
@InternalHotReloadApi  val Project.composeReloadOrchestrationPortProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.orchestration.port") 
        .orElse(providers.systemProperty("compose.reload.orchestration.port")) 
        .orElse(providers.environmentVariable("compose.reload.orchestration.port")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.OrchestrationPort]
* Current 'Orchestration Server' port. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...)
*/
@InternalHotReloadApi val Project.composeReloadOrchestrationPort: Int? get() {
    return composeReloadOrchestrationPortProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.ArgFile]
* The java 'argfile' (see: https://docs.oracle.com/en/java/javase/21/docs/specs/man/java.html#java-command-line-argument-files)
* containing all arguments of the current run
*/
@InternalHotReloadApi  val Project.composeReloadArgFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.argfile") 
        .orElse(providers.systemProperty("compose.reload.argfile")) 
        .orElse(providers.environmentVariable("compose.reload.argfile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.ArgFile]
* The java 'argfile' (see: https://docs.oracle.com/en/java/javase/21/docs/specs/man/java.html#java-command-line-argument-files)
* containing all arguments of the current run
*/
@InternalHotReloadApi val Project.composeReloadArgFile: Path? get() {
    return composeReloadArgFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.IsHeadless]
* Indicating whether or not the application is supposed to run in headless mode
*/
@InternalHotReloadApi  val Project.composeReloadIsHeadlessProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.headless") 
        .orElse(providers.systemProperty("compose.reload.headless")) 
        .orElse(providers.environmentVariable("compose.reload.headless")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IsHeadless]
* Indicating whether or not the application is supposed to run in headless mode
*/
@InternalHotReloadApi val Project.composeReloadIsHeadless: Boolean get() {
    return composeReloadIsHeadlessProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IsHotReloadBuild]
* Available in the build (e.g. Gradle) to indicate whether or not the current build is intended
* to recompile classes for a hot reload build. Typically, this property is used to disable tasks
* which are not required to produce classes and improve build speed.
*/
@InternalHotReloadApi  val Project.composeReloadIsHotReloadBuildProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.isHotReloadBuild") 
        .orElse(providers.systemProperty("compose.reload.isHotReloadBuild")) 
        .orElse(providers.environmentVariable("compose.reload.isHotReloadBuild")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IsHotReloadBuild]
* Available in the build (e.g. Gradle) to indicate whether or not the current build is intended
* to recompile classes for a hot reload build. Typically, this property is used to disable tasks
* which are not required to produce classes and improve build speed.
*/
@InternalHotReloadApi val Project.composeReloadIsHotReloadBuild: Boolean get() {
    return composeReloadIsHotReloadBuildProvider 
        .get() 
}


  /**
* See [HotReloadProperty.VirtualMethodResolveEnabled]
* true: Enable dependency analysis for virtual calls:
* e.g. Interfaces and their corresponding implementations will be tracked
*/
@InternalHotReloadApi  val Project.composeReloadVirtualMethodResolveEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.virtualMethodResolveEnabled") 
        .orElse(providers.systemProperty("compose.reload.virtualMethodResolveEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.virtualMethodResolveEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.VirtualMethodResolveEnabled]
* true: Enable dependency analysis for virtual calls:
* e.g. Interfaces and their corresponding implementations will be tracked
*/
@InternalHotReloadApi val Project.composeReloadVirtualMethodResolveEnabled: Boolean get() {
    return composeReloadVirtualMethodResolveEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DirtyResolveDepthLimit]
* If a given scope is marked as dirty, then Hot Reload will follow the dependency graph between fields/methods
* and further marks scopes as dirty. This property limits how deep this graph can be traversed.
*/
@InternalHotReloadApi  val Project.composeReloadDirtyResolveDepthLimitProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.dirtyResolveDepthLimit") 
        .orElse(providers.systemProperty("compose.reload.dirtyResolveDepthLimit")) 
        .orElse(providers.environmentVariable("compose.reload.dirtyResolveDepthLimit")) 
        .orElse("5") 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.DirtyResolveDepthLimit]
* If a given scope is marked as dirty, then Hot Reload will follow the dependency graph between fields/methods
* and further marks scopes as dirty. This property limits how deep this graph can be traversed.
*/
@InternalHotReloadApi val Project.composeReloadDirtyResolveDepthLimit: Int get() {
    return composeReloadDirtyResolveDepthLimitProvider 
        .get() 
}


  /**
* See [HotReloadProperty.GradleBuildContinuous]
* - true: Compose Hot Reload will start a recompiler Gradle Daemon, which will continuously rebuilt/reload the project
* by watching all  inputs to the build
* - false: The user is expected to rebuild/reload manually by launching a task (or using tooling)
* Continuous mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi  val Project.composeReloadGradleBuildContinuousProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.build.continuous") 
        .orElse(providers.systemProperty("compose.reload.build.continuous")) 
        .orElse(providers.environmentVariable("compose.reload.build.continuous")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.GradleBuildContinuous]
* - true: Compose Hot Reload will start a recompiler Gradle Daemon, which will continuously rebuilt/reload the project
* by watching all  inputs to the build
* - false: The user is expected to rebuild/reload manually by launching a task (or using tooling)
* Continuous mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi val Project.composeReloadGradleBuildContinuous: Boolean get() {
    return composeReloadGradleBuildContinuousProvider 
        .get() 
}


  /**
* See [HotReloadProperty.GradleWarmupEnabled]
* - true: Compose Hot Reload will launch a warmup recompile request when the application is started, to ensure that
* the recompiler Gradle daemon is running and ready to handle requests
* - false: No warmup request will be sent, first reload request may take longer time
*/
@DelicateHotReloadApi  val Project.composeReloadGradleWarmupEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.build.warmup") 
        .orElse(providers.systemProperty("compose.reload.build.warmup")) 
        .orElse(providers.environmentVariable("compose.reload.build.warmup")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.GradleWarmupEnabled]
* - true: Compose Hot Reload will launch a warmup recompile request when the application is started, to ensure that
* the recompiler Gradle daemon is running and ready to handle requests
* - false: No warmup request will be sent, first reload request may take longer time
*/
@DelicateHotReloadApi val Project.composeReloadGradleWarmupEnabled: Boolean get() {
    return composeReloadGradleWarmupEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.GradleBuildOptimize]
* - true: Compose Hot Reload will try to optimize your build during hot reload 
* (e.g. by enabling Gradle's configuration cache during 'recompilation')
* - false: No optimization will be performed.
*/
@DelicateHotReloadApi  val Project.composeReloadGradleBuildOptimizeProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.gradle.build.optimize") 
        .orElse(providers.systemProperty("compose.reload.gradle.build.optimize")) 
        .orElse(providers.environmentVariable("compose.reload.gradle.build.optimize")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.GradleBuildOptimize]
* - true: Compose Hot Reload will try to optimize your build during hot reload 
* (e.g. by enabling Gradle's configuration cache during 'recompilation')
* - false: No optimization will be performed.
*/
@DelicateHotReloadApi val Project.composeReloadGradleBuildOptimize: Boolean get() {
    return composeReloadGradleBuildOptimizeProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsEnabled]
* Flag to disable the 'devtools' application entirely
*/
 val Project.composeReloadDevToolsEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsEnabled") 
        .orElse(providers.systemProperty("compose.reload.devToolsEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsEnabled]
* Flag to disable the 'devtools' application entirely
*/
val Project.composeReloadDevToolsEnabled: Boolean get() {
    return composeReloadDevToolsEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsIsHeadless]
* Run the dev tools in headless mode (No UI window shown)
*/
 val Project.composeReloadDevToolsIsHeadlessProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsHeadless") 
        .orElse(providers.systemProperty("compose.reload.devToolsHeadless")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsHeadless")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsIsHeadless]
* Run the dev tools in headless mode (No UI window shown)
*/
val Project.composeReloadDevToolsIsHeadless: Boolean get() {
    return composeReloadDevToolsIsHeadlessProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsTransparencyEnabled]
* Some platforms might not be able to render transparency correctly (e.g. some linux environments).
* This property will allow such platforms to disable/enable transparency. This property is subject to change
* if the issues with transparency rendering are resolved..
*/
@DelicateHotReloadApi  val Project.composeReloadDevToolsTransparencyEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsTransparencyEnabled") 
        .orElse(providers.systemProperty("compose.reload.devToolsTransparencyEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsTransparencyEnabled")) 
        .orElse((Os.currentOrNull() != Os.Linux).toString()) 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsTransparencyEnabled]
* Some platforms might not be able to render transparency correctly (e.g. some linux environments).
* This property will allow such platforms to disable/enable transparency. This property is subject to change
* if the issues with transparency rendering are resolved..
*/
@DelicateHotReloadApi val Project.composeReloadDevToolsTransparencyEnabled: Boolean get() {
    return composeReloadDevToolsTransparencyEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsDetached]
* If enabled, dev tools window will be detached from the main application
*/
 val Project.composeReloadDevToolsDetachedProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsDetached") 
        .orElse(providers.systemProperty("compose.reload.devToolsDetached")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsDetached")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsDetached]
* If enabled, dev tools window will be detached from the main application
*/
val Project.composeReloadDevToolsDetached: Boolean get() {
    return composeReloadDevToolsDetachedProvider 
        .get() 
}


  /**
* See [HotReloadProperty.ReloadEffectsEnabled]
* Enable reload effects that are shown in the UI when a reload is triggered.
*/
 val Project.composeReloadReloadEffectsEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.effectsEnabled") 
        .orElse(providers.systemProperty("compose.reload.effectsEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.effectsEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.ReloadEffectsEnabled]
* Enable reload effects that are shown in the UI when a reload is triggered.
*/
val Project.composeReloadReloadEffectsEnabled: Boolean get() {
    return composeReloadReloadEffectsEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IntelliJDebuggerDispatchPort]
* Note: Expected as an environment variable, as this is expected to be transitively available
* to all child processes.\n
* Currently, launching applications with hot reload might be done through a couple of
* intermediate processes. For example, launching a test will go through a chain like
* ```
* intellij --launches--> Gradle --launches--> JVM(Junit) --launches--> Gradle
* --launches--> JVM (Application)
* ```
* When a run configuration is started in 'debug mode' intellij will set the system property
* 'idea.debugger.dispatch.port'. This will indicate that a server is listening at this port, which can
* be used to provision debugging servers.
* This debug port will then be made available as an environment variable using this key.
* Launching the final application will respect this port, if present and provision a debugging session.
* This will allow a test to be deeply debuggable by just pressing 'Debug'
*/
@InternalHotReloadApi  val Project.composeReloadIntelliJDebuggerDispatchPortProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.idea.debugger.dispatch.port") 
        .orElse(providers.systemProperty("compose.reload.idea.debugger.dispatch.port")) 
        .orElse(providers.environmentVariable("compose.reload.idea.debugger.dispatch.port")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.IntelliJDebuggerDispatchPort]
* Note: Expected as an environment variable, as this is expected to be transitively available
* to all child processes.\n
* Currently, launching applications with hot reload might be done through a couple of
* intermediate processes. For example, launching a test will go through a chain like
* ```
* intellij --launches--> Gradle --launches--> JVM(Junit) --launches--> Gradle
* --launches--> JVM (Application)
* ```
* When a run configuration is started in 'debug mode' intellij will set the system property
* 'idea.debugger.dispatch.port'. This will indicate that a server is listening at this port, which can
* be used to provision debugging servers.
* This debug port will then be made available as an environment variable using this key.
* Launching the final application will respect this port, if present and provision a debugging session.
* This will allow a test to be deeply debuggable by just pressing 'Debug'
*/
@InternalHotReloadApi val Project.composeReloadIntelliJDebuggerDispatchPort: Int? get() {
    return composeReloadIntelliJDebuggerDispatchPortProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.SubprocessDebuggingEnabled]
* Enable this property to allow propagating the 'idea.debugger.dispatch.port' to all subprocesses.
* This is useful when debugging dev tools. Note: this may break the debugging of the user application if IJ is
* not configured to accept multiple debugging sessions. Not recommended to use outside of hot reload debugging.
*/
@InternalHotReloadApi  val Project.composeReloadSubprocessDebuggingEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.subprocessDebuggingEnabled") 
        .orElse(providers.systemProperty("compose.reload.subprocessDebuggingEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.subprocessDebuggingEnabled")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.SubprocessDebuggingEnabled]
* Enable this property to allow propagating the 'idea.debugger.dispatch.port' to all subprocesses.
* This is useful when debugging dev tools. Note: this may break the debugging of the user application if IJ is
* not configured to accept multiple debugging sessions. Not recommended to use outside of hot reload debugging.
*/
@InternalHotReloadApi val Project.composeReloadSubprocessDebuggingEnabled: Boolean get() {
    return composeReloadSubprocessDebuggingEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.JetBrainsRuntimeBinary]
* The path to the 'JetBrainsRuntime' which shall be used when launching the app. 
* Note: This is a build-only property!
*/
@DelicateHotReloadApi  val Project.composeReloadJetBrainsRuntimeBinaryProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.jbr.binary") 
        .orElse(providers.systemProperty("compose.reload.jbr.binary")) 
        .orElse(providers.environmentVariable("compose.reload.jbr.binary")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.JetBrainsRuntimeBinary]
* The path to the 'JetBrainsRuntime' which shall be used when launching the app. 
* Note: This is a build-only property!
*/
@DelicateHotReloadApi val Project.composeReloadJetBrainsRuntimeBinary: Path? get() {
    return composeReloadJetBrainsRuntimeBinaryProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.JetBrainsRuntimeVersion]
* Specifies the default 'JetBrains Runtime' version that shall be used (e.g. '21' or '25')
*/
@DelicateHotReloadApi  val Project.composeReloadJetBrainsRuntimeVersionProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.jbr.version") 
        .orElse(providers.systemProperty("compose.reload.jbr.version")) 
        .orElse(providers.environmentVariable("compose.reload.jbr.version")) 
        .orElse("21") 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.JetBrainsRuntimeVersion]
* Specifies the default 'JetBrains Runtime' version that shall be used (e.g. '21' or '25')
*/
@DelicateHotReloadApi val Project.composeReloadJetBrainsRuntimeVersion: Int get() {
    return composeReloadJetBrainsRuntimeVersionProvider 
        .get() 
}


  /**
* See [HotReloadProperty.AutoRuntimeDependenciesEnabled]
* Whether or not the hot-reload-runtime will be added as a compile dependency automatically when running a build.
*/
@InternalHotReloadApi  val Project.composeReloadAutoRuntimeDependenciesEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.autoRuntimeDependenciesEnabled") 
        .orElse(providers.systemProperty("compose.reload.autoRuntimeDependenciesEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.autoRuntimeDependenciesEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.AutoRuntimeDependenciesEnabled]
* Whether or not the hot-reload-runtime will be added as a compile dependency automatically when running a build.
*/
@InternalHotReloadApi val Project.composeReloadAutoRuntimeDependenciesEnabled: Boolean get() {
    return composeReloadAutoRuntimeDependenciesEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IdeaComposeHotReload]
* Set by IntelliJ to signal the Gradle Plugin that IDE tooling is available.
* Setting this variable will relax the Gradle Plugin to not touch existing run tasks as we expect
* the IDE to provide a convenient way of launching in hot-reload mode.
*/
@InternalHotReloadApi  val Project.composeReloadIdeaComposeHotReloadProvider: Provider<Boolean> get() {
    return providers.gradleProperty("idea.compose.hot-reload") 
        .orElse(providers.systemProperty("idea.compose.hot-reload")) 
        .orElse(providers.environmentVariable("idea.compose.hot-reload")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IdeaComposeHotReload]
* Set by IntelliJ to signal the Gradle Plugin that IDE tooling is available.
* Setting this variable will relax the Gradle Plugin to not touch existing run tasks as we expect
* the IDE to provide a convenient way of launching in hot-reload mode.
*/
@InternalHotReloadApi val Project.composeReloadIdeaComposeHotReload: Boolean get() {
    return composeReloadIdeaComposeHotReloadProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IdeaComposeHotReloadSupportVersion]
* Set by IntelliJ during builds to convey its 'support level' for hot reload.
* Not Present, but 'idea.compose.hot-reload' being set: Supports running hot run tasks
* 2: Support running 'hot run' tasks and 'hot reload' tasks
* 3: Supports providing IDEA runtime using 'idea.compose.hot-reload.jbr'
*/
@InternalHotReloadApi  val Project.composeReloadIdeaComposeHotReloadSupportVersionProvider: Provider<Int> get() {
    return providers.gradleProperty("idea.compose.hot-reload.version") 
        .orElse(providers.systemProperty("idea.compose.hot-reload.version")) 
        .orElse(providers.environmentVariable("idea.compose.hot-reload.version")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.IdeaComposeHotReloadSupportVersion]
* Set by IntelliJ during builds to convey its 'support level' for hot reload.
* Not Present, but 'idea.compose.hot-reload' being set: Supports running hot run tasks
* 2: Support running 'hot run' tasks and 'hot reload' tasks
* 3: Supports providing IDEA runtime using 'idea.compose.hot-reload.jbr'
*/
@InternalHotReloadApi val Project.composeReloadIdeaComposeHotReloadSupportVersion: Int? get() {
    return composeReloadIdeaComposeHotReloadSupportVersionProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.IdeaJetBrainsRuntimeBinary]
* Forwards the 'JetBrains Runtime' which is bundled with the current version of IntelliJ
* - (Points to the binary (bin/java))
* - Can be used as 'fallback' when no other JBR is found
*/
@InternalHotReloadApi  val Project.composeReloadIdeaJetBrainsRuntimeBinaryProvider: Provider<Path> get() {
    return providers.gradleProperty("idea.compose.hot-reload.jbr") 
        .orElse(providers.systemProperty("idea.compose.hot-reload.jbr")) 
        .orElse(providers.environmentVariable("idea.compose.hot-reload.jbr")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.IdeaJetBrainsRuntimeBinary]
* Forwards the 'JetBrains Runtime' which is bundled with the current version of IntelliJ
* - (Points to the binary (bin/java))
* - Can be used as 'fallback' when no other JBR is found
*/
@InternalHotReloadApi val Project.composeReloadIdeaJetBrainsRuntimeBinary: Path? get() {
    return composeReloadIdeaJetBrainsRuntimeBinaryProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StdinFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to the stdin file (can be pipe)
* 'async'/'non-blocking' mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi  val Project.composeReloadStdinFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stdinFile") 
        .orElse(providers.systemProperty("compose.reload.stdinFile")) 
        .orElse(providers.environmentVariable("compose.reload.stdinFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StdinFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to the stdin file (can be pipe)
* 'async'/'non-blocking' mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi val Project.composeReloadStdinFile: Path? get() {
    return composeReloadStdinFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StdoutFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stdout is supposed to be written to.
* 'async'/'non-blocking' mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi  val Project.composeReloadStdoutFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stdoutFile") 
        .orElse(providers.systemProperty("compose.reload.stdoutFile")) 
        .orElse(providers.environmentVariable("compose.reload.stdoutFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StdoutFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stdout is supposed to be written to.
* 'async'/'non-blocking' mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi val Project.composeReloadStdoutFile: Path? get() {
    return composeReloadStdoutFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StderrFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stderr is supposed to be written to.
* 'async'/'non-blocking' mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi  val Project.composeReloadStderrFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stderrFile") 
        .orElse(providers.systemProperty("compose.reload.stderrFile")) 
        .orElse(providers.environmentVariable("compose.reload.stderrFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StderrFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stderr is supposed to be written to.
* 'async'/'non-blocking' mode is subject to change and might be removed in the future.
*/
@DelicateHotReloadApi val Project.composeReloadStderrFile: Path? get() {
    return composeReloadStderrFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.ParentPid]
* The pid of the parent process. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...).
* If the parent dies, it is expected that children exit as well.
*/
@InternalHotReloadApi  val Project.composeReloadParentPidProvider: Provider<Long> get() {
    return providers.gradleProperty("compose.reload.parentPid") 
        .orElse(providers.systemProperty("compose.reload.parentPid")) 
        .orElse(providers.environmentVariable("compose.reload.parentPid")) 
        .map { raw -> raw.toLong() } 
}

/**
* See [HotReloadProperty.ParentPid]
* The pid of the parent process. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...).
* If the parent dies, it is expected that children exit as well.
*/
@InternalHotReloadApi val Project.composeReloadParentPid: Long? get() {
    return composeReloadParentPidProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.LogLevel]
* Minimum logging level
*/
 val Project.composeReloadLogLevelProvider: Provider<org.jetbrains.compose.reload.core.Logger.Level> get() {
    return providers.gradleProperty("compose.reload.logLevel") 
        .orElse(providers.systemProperty("compose.reload.logLevel")) 
        .orElse(providers.environmentVariable("compose.reload.logLevel")) 
        .orElse("Info") 
        .map { raw -> enumValueOf<org.jetbrains.compose.reload.core.Logger.Level>(raw) } 
}

/**
* See [HotReloadProperty.LogLevel]
* Minimum logging level
*/
val Project.composeReloadLogLevel: org.jetbrains.compose.reload.core.Logger.Level get() {
    return composeReloadLogLevelProvider 
        .get() 
}


  /**
* See [HotReloadProperty.LogStdout]
* Enable output of all logs into the standard output
*/
 val Project.composeReloadLogStdoutProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.logStdout") 
        .orElse(providers.systemProperty("compose.reload.logStdout")) 
        .orElse(providers.environmentVariable("compose.reload.logStdout")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.LogStdout]
* Enable output of all logs into the standard output
*/
val Project.composeReloadLogStdout: Boolean get() {
    return composeReloadLogStdoutProvider 
        .get() 
}
