package org.jetbrains.compose.reload.gradle
import java.nio.file.Path
import kotlin.io.path.Path
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.jetbrains.compose.reload.core.HotReloadProperty
import org.jetbrains.compose.reload.core.Os
import org.jetbrains.compose.reload.InternalHotReloadApi


  /**
* See [HotReloadProperty.OrchestrationPort]
* Current 'Orchestration Server' port. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...)
*/
@InternalHotReloadApi val Project.composeReloadOrchestrationPortProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.orchestration.port") 
        .orElse(providers.systemProperty("compose.reload.orchestration.port")) 
        .orElse(providers.environmentVariable("compose.reload.orchestration.port")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.OrchestrationPort]
* Current 'Orchestration Server' port. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...)
*/
@InternalHotReloadApi val Project.composeReloadOrchestrationPort: Int? get() {
    return composeReloadOrchestrationPortProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.ArgFile]
* The java 'argfile' (see: https://docs.oracle.com/en/java/javase/21/docs/specs/man/java.html#java-command-line-argument-files)
* containing all arguments of the current run
*/
@InternalHotReloadApi val Project.composeReloadArgFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.argfile") 
        .orElse(providers.systemProperty("compose.reload.argfile")) 
        .orElse(providers.environmentVariable("compose.reload.argfile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.ArgFile]
* The java 'argfile' (see: https://docs.oracle.com/en/java/javase/21/docs/specs/man/java.html#java-command-line-argument-files)
* containing all arguments of the current run
*/
@InternalHotReloadApi val Project.composeReloadArgFile: Path? get() {
    return composeReloadArgFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.IsHeadless]
* Indicating whether or not the application is supposed to run in headless mode
*/
@InternalHotReloadApi val Project.composeReloadIsHeadlessProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.headless") 
        .orElse(providers.systemProperty("compose.reload.headless")) 
        .orElse(providers.environmentVariable("compose.reload.headless")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IsHeadless]
* Indicating whether or not the application is supposed to run in headless mode
*/
@InternalHotReloadApi val Project.composeReloadIsHeadless: Boolean get() {
    return composeReloadIsHeadlessProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IsHotReloadBuild]
* Available in the build (e.g. Gradle) to indicate whether or not the current build is intended
* to recompile classes for a hot reload build. Typically, this property is used to disable tasks
* which are not required to produce classes and improve build speed.
*/
@InternalHotReloadApi val Project.composeReloadIsHotReloadBuildProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.isHotReloadBuild") 
        .orElse(providers.systemProperty("compose.reload.isHotReloadBuild")) 
        .orElse(providers.environmentVariable("compose.reload.isHotReloadBuild")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IsHotReloadBuild]
* Available in the build (e.g. Gradle) to indicate whether or not the current build is intended
* to recompile classes for a hot reload build. Typically, this property is used to disable tasks
* which are not required to produce classes and improve build speed.
*/
@InternalHotReloadApi val Project.composeReloadIsHotReloadBuild: Boolean get() {
    return composeReloadIsHotReloadBuildProvider 
        .get() 
}


  /**
* See [HotReloadProperty.VirtualMethodResolveEnabled]
* true: Enable dependency analysis for virtual calls:
* e.g. Interfaces and their corresponding implementations will be tracked
*/
@InternalHotReloadApi val Project.composeReloadVirtualMethodResolveEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.virtualMethodResolveEnabled") 
        .orElse(providers.systemProperty("compose.reload.virtualMethodResolveEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.virtualMethodResolveEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.VirtualMethodResolveEnabled]
* true: Enable dependency analysis for virtual calls:
* e.g. Interfaces and their corresponding implementations will be tracked
*/
@InternalHotReloadApi val Project.composeReloadVirtualMethodResolveEnabled: Boolean get() {
    return composeReloadVirtualMethodResolveEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DirtyResolveDepthLimit]
* If a given scope is marked as dirty, then Hot Reload will follow the dependency graph between fields/methods
* and further marks scopes as dirty. This property limits how deep this graph can be traversed.
*/
@InternalHotReloadApi val Project.composeReloadDirtyResolveDepthLimitProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.dirtyResolveDepthLimit") 
        .orElse(providers.systemProperty("compose.reload.dirtyResolveDepthLimit")) 
        .orElse(providers.environmentVariable("compose.reload.dirtyResolveDepthLimit")) 
        .orElse("5") 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.DirtyResolveDepthLimit]
* If a given scope is marked as dirty, then Hot Reload will follow the dependency graph between fields/methods
* and further marks scopes as dirty. This property limits how deep this graph can be traversed.
*/
@InternalHotReloadApi val Project.composeReloadDirtyResolveDepthLimit: Int get() {
    return composeReloadDirtyResolveDepthLimitProvider 
        .get() 
}


  /**
* See [HotReloadProperty.GradleBuildContinuous]
* - true: Compose Hot Reload will start a recompiler Gradle Daemon, which will continuously rebuilt/reload the project
* by watching all  inputs to the build
* - false: The user is expected to rebuild/reload manually by launching a task (or using tooling)
*/
@InternalHotReloadApi val Project.composeReloadGradleBuildContinuousProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.build.continuous") 
        .orElse(providers.systemProperty("compose.reload.build.continuous")) 
        .orElse(providers.environmentVariable("compose.reload.build.continuous")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.GradleBuildContinuous]
* - true: Compose Hot Reload will start a recompiler Gradle Daemon, which will continuously rebuilt/reload the project
* by watching all  inputs to the build
* - false: The user is expected to rebuild/reload manually by launching a task (or using tooling)
*/
@InternalHotReloadApi val Project.composeReloadGradleBuildContinuous: Boolean get() {
    return composeReloadGradleBuildContinuousProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsEnabled]
* Flag to disable the 'devtools' application entirely
*/
@InternalHotReloadApi val Project.composeReloadDevToolsEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsEnabled") 
        .orElse(providers.systemProperty("compose.reload.devToolsEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsEnabled]
* Flag to disable the 'devtools' application entirely
*/
@InternalHotReloadApi val Project.composeReloadDevToolsEnabled: Boolean get() {
    return composeReloadDevToolsEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsIsHeadless]
* Run the dev tools in headless mode (No UI window shown)
*/
@InternalHotReloadApi val Project.composeReloadDevToolsIsHeadlessProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsHeadless") 
        .orElse(providers.systemProperty("compose.reload.devToolsHeadless")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsHeadless")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsIsHeadless]
* Run the dev tools in headless mode (No UI window shown)
*/
@InternalHotReloadApi val Project.composeReloadDevToolsIsHeadless: Boolean get() {
    return composeReloadDevToolsIsHeadlessProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsTransparencyEnabled]
* Some platforms might not be able to render transparency correctly (e.g. some linux environments).
* This property will allow such platforms to disable/enable transparency
*/
@InternalHotReloadApi val Project.composeReloadDevToolsTransparencyEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsTransparencyEnabled") 
        .orElse(providers.systemProperty("compose.reload.devToolsTransparencyEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsTransparencyEnabled")) 
        .orElse((Os.currentOrNull() != Os.Linux).toString()) 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsTransparencyEnabled]
* Some platforms might not be able to render transparency correctly (e.g. some linux environments).
* This property will allow such platforms to disable/enable transparency
*/
@InternalHotReloadApi val Project.composeReloadDevToolsTransparencyEnabled: Boolean get() {
    return composeReloadDevToolsTransparencyEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsDetached]
* If enabled, dev tools window will be detached from the main application
*/
@InternalHotReloadApi val Project.composeReloadDevToolsDetachedProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsDetached") 
        .orElse(providers.systemProperty("compose.reload.devToolsDetached")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsDetached")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsDetached]
* If enabled, dev tools window will be detached from the main application
*/
@InternalHotReloadApi val Project.composeReloadDevToolsDetached: Boolean get() {
    return composeReloadDevToolsDetachedProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IntelliJDebuggerDispatchPort]
* Note: Expected as an environment variable, as this is expected to be transitively available
* to all child processes.\n
* Currently, launching applications with hot reload might be done through a couple of
* intermediate processes. For example, launching a test will go through a chain like
* ```
* intellij --launches--> Gradle --launches--> JVM(Junit) --launches--> Gradle
* --launches--> JVM (Application)
* ```
* When a run configuration is started in 'debug mode' intellij will set the system property
* 'idea.debugger.dispatch.port'. This will indicate that a server is listening at this port, which can
* be used to provision debugging servers.
* This debug port will then be made available as an environment variable using this key.
* Launching the final application will respect this port, if present and provision a debugging session.
* This will allow a test to be deeply debuggable by just pressing 'Debug'
*/
@InternalHotReloadApi val Project.composeReloadIntelliJDebuggerDispatchPortProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.idea.debugger.dispatch.port") 
        .orElse(providers.systemProperty("compose.reload.idea.debugger.dispatch.port")) 
        .orElse(providers.environmentVariable("compose.reload.idea.debugger.dispatch.port")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.IntelliJDebuggerDispatchPort]
* Note: Expected as an environment variable, as this is expected to be transitively available
* to all child processes.\n
* Currently, launching applications with hot reload might be done through a couple of
* intermediate processes. For example, launching a test will go through a chain like
* ```
* intellij --launches--> Gradle --launches--> JVM(Junit) --launches--> Gradle
* --launches--> JVM (Application)
* ```
* When a run configuration is started in 'debug mode' intellij will set the system property
* 'idea.debugger.dispatch.port'. This will indicate that a server is listening at this port, which can
* be used to provision debugging servers.
* This debug port will then be made available as an environment variable using this key.
* Launching the final application will respect this port, if present and provision a debugging session.
* This will allow a test to be deeply debuggable by just pressing 'Debug'
*/
@InternalHotReloadApi val Project.composeReloadIntelliJDebuggerDispatchPort: Int? get() {
    return composeReloadIntelliJDebuggerDispatchPortProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.JetBrainsRuntimeBinary]
* The path to the 'JetBrainsRuntime' which shall be used when launching the app. 
* Note: This is a build-only property!
*/
@InternalHotReloadApi val Project.composeReloadJetBrainsRuntimeBinaryProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.jbr.binary") 
        .orElse(providers.systemProperty("compose.reload.jbr.binary")) 
        .orElse(providers.environmentVariable("compose.reload.jbr.binary")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.JetBrainsRuntimeBinary]
* The path to the 'JetBrainsRuntime' which shall be used when launching the app. 
* Note: This is a build-only property!
*/
@InternalHotReloadApi val Project.composeReloadJetBrainsRuntimeBinary: Path? get() {
    return composeReloadJetBrainsRuntimeBinaryProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.AutoRuntimeDependenciesEnabled]
* Whether or not the hot-reload-runtime will be added as a compile dependency automatically when running a build.
*/
@InternalHotReloadApi val Project.composeReloadAutoRuntimeDependenciesEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.autoRuntimeDependenciesEnabled") 
        .orElse(providers.systemProperty("compose.reload.autoRuntimeDependenciesEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.autoRuntimeDependenciesEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.AutoRuntimeDependenciesEnabled]
* Whether or not the hot-reload-runtime will be added as a compile dependency automatically when running a build.
*/
@InternalHotReloadApi val Project.composeReloadAutoRuntimeDependenciesEnabled: Boolean get() {
    return composeReloadAutoRuntimeDependenciesEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IdeaComposeHotReload]
* Set by IntelliJ to signal the Gradle Plugin that IDE tooling is available.
* Setting this variable will relax the Gradle Plugin to not touch existing run tasks as we expect
* the IDE to provide a convenient way of launching in hot-reload mode.
*/
@InternalHotReloadApi val Project.composeReloadIdeaComposeHotReloadProvider: Provider<Boolean> get() {
    return providers.gradleProperty("idea.compose.hot-reload") 
        .orElse(providers.systemProperty("idea.compose.hot-reload")) 
        .orElse(providers.environmentVariable("idea.compose.hot-reload")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IdeaComposeHotReload]
* Set by IntelliJ to signal the Gradle Plugin that IDE tooling is available.
* Setting this variable will relax the Gradle Plugin to not touch existing run tasks as we expect
* the IDE to provide a convenient way of launching in hot-reload mode.
*/
@InternalHotReloadApi val Project.composeReloadIdeaComposeHotReload: Boolean get() {
    return composeReloadIdeaComposeHotReloadProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IdeaComposeHotReloadSupportVersion]
* Set by IntelliJ during builds to convey its 'support level' for hot reload.
* Not Present, but 'idea.compose.hot-reload' being set: Supports running hot run tasks
* 
* 2: Support running 'hot run' tasks and 'hot reload' tasks
*/
@InternalHotReloadApi val Project.composeReloadIdeaComposeHotReloadSupportVersionProvider: Provider<Int> get() {
    return providers.gradleProperty("idea.compose.hot-reload.version") 
        .orElse(providers.systemProperty("idea.compose.hot-reload.version")) 
        .orElse(providers.environmentVariable("idea.compose.hot-reload.version")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.IdeaComposeHotReloadSupportVersion]
* Set by IntelliJ during builds to convey its 'support level' for hot reload.
* Not Present, but 'idea.compose.hot-reload' being set: Supports running hot run tasks
* 
* 2: Support running 'hot run' tasks and 'hot reload' tasks
*/
@InternalHotReloadApi val Project.composeReloadIdeaComposeHotReloadSupportVersion: Int? get() {
    return composeReloadIdeaComposeHotReloadSupportVersionProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StdinFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to the stdin file (can be pipe)
*/
@InternalHotReloadApi val Project.composeReloadStdinFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stdinFile") 
        .orElse(providers.systemProperty("compose.reload.stdinFile")) 
        .orElse(providers.environmentVariable("compose.reload.stdinFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StdinFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to the stdin file (can be pipe)
*/
@InternalHotReloadApi val Project.composeReloadStdinFile: Path? get() {
    return composeReloadStdinFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StdoutFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stdout is supposed to be written to.
*/
@InternalHotReloadApi val Project.composeReloadStdoutFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stdoutFile") 
        .orElse(providers.systemProperty("compose.reload.stdoutFile")) 
        .orElse(providers.environmentVariable("compose.reload.stdoutFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StdoutFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stdout is supposed to be written to.
*/
@InternalHotReloadApi val Project.composeReloadStdoutFile: Path? get() {
    return composeReloadStdoutFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StderrFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stderr is supposed to be written to.
*/
@InternalHotReloadApi val Project.composeReloadStderrFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stderrFile") 
        .orElse(providers.systemProperty("compose.reload.stderrFile")) 
        .orElse(providers.environmentVariable("compose.reload.stderrFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StderrFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stderr is supposed to be written to.
*/
@InternalHotReloadApi val Project.composeReloadStderrFile: Path? get() {
    return composeReloadStderrFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.ParentPid]
* The pid of the parent process. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...).
* If the parent dies, it is expected that children exit as well.
*/
@InternalHotReloadApi val Project.composeReloadParentPidProvider: Provider<Long> get() {
    return providers.gradleProperty("compose.reload.parentPid") 
        .orElse(providers.systemProperty("compose.reload.parentPid")) 
        .orElse(providers.environmentVariable("compose.reload.parentPid")) 
        .map { raw -> raw.toLong() } 
}

/**
* See [HotReloadProperty.ParentPid]
* The pid of the parent process. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...).
* If the parent dies, it is expected that children exit as well.
*/
@InternalHotReloadApi val Project.composeReloadParentPid: Long? get() {
    return composeReloadParentPidProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.LogLevel]
* Minimum logging level
*/
@InternalHotReloadApi val Project.composeReloadLogLevelProvider: Provider<org.jetbrains.compose.reload.core.Logger.Level> get() {
    return providers.gradleProperty("compose.reload.logLevel") 
        .orElse(providers.systemProperty("compose.reload.logLevel")) 
        .orElse(providers.environmentVariable("compose.reload.logLevel")) 
        .orElse("Info") 
        .map { raw -> enumValueOf<org.jetbrains.compose.reload.core.Logger.Level>(raw) } 
}

/**
* See [HotReloadProperty.LogLevel]
* Minimum logging level
*/
@InternalHotReloadApi val Project.composeReloadLogLevel: org.jetbrains.compose.reload.core.Logger.Level get() {
    return composeReloadLogLevelProvider 
        .get() 
}


  /**
* See [HotReloadProperty.LogStdout]
* Enable output of all logs into the standard output
*/
@InternalHotReloadApi val Project.composeReloadLogStdoutProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.logStdout") 
        .orElse(providers.systemProperty("compose.reload.logStdout")) 
        .orElse(providers.environmentVariable("compose.reload.logStdout")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.LogStdout]
* Enable output of all logs into the standard output
*/
@InternalHotReloadApi val Project.composeReloadLogStdout: Boolean get() {
    return composeReloadLogStdoutProvider 
        .get() 
}


  /**
* See [HotReloadProperty.ResourcesDirtyResolverEnabled]
* Enables/Disables automatically re-composing scopes using resources
* See: https://www.jetbrains.com/help/kotlin-multiplatform-dev/compose-multiplatform-resources.html
* See: https://github.com/JetBrains/compose-hot-reload/issues/151
*/
@InternalHotReloadApi val Project.composeReloadResourcesDirtyResolverEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.resourcesDirtyResolverEnabled") 
        .orElse(providers.systemProperty("compose.reload.resourcesDirtyResolverEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.resourcesDirtyResolverEnabled")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.ResourcesDirtyResolverEnabled]
* Enables/Disables automatically re-composing scopes using resources
* See: https://www.jetbrains.com/help/kotlin-multiplatform-dev/compose-multiplatform-resources.html
* See: https://github.com/JetBrains/compose-hot-reload/issues/151
*/
@InternalHotReloadApi val Project.composeReloadResourcesDirtyResolverEnabled: Boolean get() {
    return composeReloadResourcesDirtyResolverEnabledProvider 
        .get() 
}
