package org.jetbrains.compose.reload.gradle.core
import java.nio.file.Path
import kotlin.io.path.Path
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.jetbrains.compose.reload.core.HotReloadProperty
import org.jetbrains.compose.reload.core.Os
import org.jetbrains.compose.reload.gradle.InternalHotReloadGradleApi


  /**
* See [HotReloadProperty.OrchestrationPort]
* Current 'Orchestration Server' port. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...)
*/
@InternalHotReloadGradleApi val Project.composeReloadOrchestrationPortProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.orchestration.port") 
        .orElse(providers.systemProperty("compose.reload.orchestration.port")) 
        .orElse(providers.environmentVariable("compose.reload.orchestration.port")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.OrchestrationPort]
* Current 'Orchestration Server' port. This property is used by components that shall connect to the 
* 'orchestration' (e.g. recompiler, devtools, ...)
*/
@InternalHotReloadGradleApi val Project.composeReloadOrchestrationPort: Int? get() {
    return composeReloadOrchestrationPortProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.ArgFile]
* The java 'argfile' (see: https://docs.oracle.com/en/java/javase/21/docs/specs/man/java.html#java-command-line-argument-files)
* containing all arguments of the current run
*/
@InternalHotReloadGradleApi val Project.composeReloadArgFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.argfile") 
        .orElse(providers.systemProperty("compose.reload.argfile")) 
        .orElse(providers.environmentVariable("compose.reload.argfile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.ArgFile]
* The java 'argfile' (see: https://docs.oracle.com/en/java/javase/21/docs/specs/man/java.html#java-command-line-argument-files)
* containing all arguments of the current run
*/
@InternalHotReloadGradleApi val Project.composeReloadArgFile: Path? get() {
    return composeReloadArgFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.IsHeadless]
* Indicating whether or not the application is supposed to run in headless mode
*/
@InternalHotReloadGradleApi val Project.composeReloadIsHeadlessProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.headless") 
        .orElse(providers.systemProperty("compose.reload.headless")) 
        .orElse(providers.environmentVariable("compose.reload.headless")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IsHeadless]
* Indicating whether or not the application is supposed to run in headless mode
*/
@InternalHotReloadGradleApi val Project.composeReloadIsHeadless: Boolean get() {
    return composeReloadIsHeadlessProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IsHotReloadBuild]
* Available in the build (e.g. Gradle) to indicate whether or not the current build is intended
* to recompile classes for a hot reload build. Typically, this property is used to disable tasks
* which are not required to produce classes and improve build speed.
*/
@InternalHotReloadGradleApi val Project.composeReloadIsHotReloadBuildProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.isHotReloadBuild") 
        .orElse(providers.systemProperty("compose.reload.isHotReloadBuild")) 
        .orElse(providers.environmentVariable("compose.reload.isHotReloadBuild")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IsHotReloadBuild]
* Available in the build (e.g. Gradle) to indicate whether or not the current build is intended
* to recompile classes for a hot reload build. Typically, this property is used to disable tasks
* which are not required to produce classes and improve build speed.
*/
@InternalHotReloadGradleApi val Project.composeReloadIsHotReloadBuild: Boolean get() {
    return composeReloadIsHotReloadBuildProvider 
        .get() 
}


  /**
* See [HotReloadProperty.VirtualMethodResolveEnabled]
* true: Enable dependency analysis for virtual calls:
* e.g. Interfaces and their corresponding implementations will be tracked
*/
@InternalHotReloadGradleApi val Project.composeReloadVirtualMethodResolveEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.virtualMethodResolveEnabled") 
        .orElse(providers.systemProperty("compose.reload.virtualMethodResolveEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.virtualMethodResolveEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.VirtualMethodResolveEnabled]
* true: Enable dependency analysis for virtual calls:
* e.g. Interfaces and their corresponding implementations will be tracked
*/
@InternalHotReloadGradleApi val Project.composeReloadVirtualMethodResolveEnabled: Boolean get() {
    return composeReloadVirtualMethodResolveEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DirtyResolveDepthLimit]
* If a given scope is marked as dirty, then Hot Reload will follow the dependency graph between fields/methods
* and further marks scopes as dirty. This property limits how deep this graph can be traversed.
*/
@InternalHotReloadGradleApi val Project.composeReloadDirtyResolveDepthLimitProvider: Provider<Int> get() {
    return providers.gradleProperty("compose.reload.dirtyResolveDepthLimit") 
        .orElse(providers.systemProperty("compose.reload.dirtyResolveDepthLimit")) 
        .orElse(providers.environmentVariable("compose.reload.dirtyResolveDepthLimit")) 
        .orElse("5") 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.DirtyResolveDepthLimit]
* If a given scope is marked as dirty, then Hot Reload will follow the dependency graph between fields/methods
* and further marks scopes as dirty. This property limits how deep this graph can be traversed.
*/
@InternalHotReloadGradleApi val Project.composeReloadDirtyResolveDepthLimit: Int get() {
    return composeReloadDirtyResolveDepthLimitProvider 
        .get() 
}


  /**
* See [HotReloadProperty.GradleBuildContinuous]
* - true: Compose Hot Reload will start a recompiler Gradle Daemon, which will continuously rebuilt/reload the project
* by watching all  inputs to the build
* - false: The user is expected to rebuild/reload manually by launching a task (or using tooling)
*/
@InternalHotReloadGradleApi val Project.composeReloadGradleBuildContinuousProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.build.continuous") 
        .orElse(providers.systemProperty("compose.reload.build.continuous")) 
        .orElse(providers.environmentVariable("compose.reload.build.continuous")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.GradleBuildContinuous]
* - true: Compose Hot Reload will start a recompiler Gradle Daemon, which will continuously rebuilt/reload the project
* by watching all  inputs to the build
* - false: The user is expected to rebuild/reload manually by launching a task (or using tooling)
*/
@InternalHotReloadGradleApi val Project.composeReloadGradleBuildContinuous: Boolean get() {
    return composeReloadGradleBuildContinuousProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsEnabled]
* Flag to disable the 'devtools' application entirely
*/
@InternalHotReloadGradleApi val Project.composeReloadDevToolsEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsEnabled") 
        .orElse(providers.systemProperty("compose.reload.devToolsEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsEnabled]
* Flag to disable the 'devtools' application entirely
*/
@InternalHotReloadGradleApi val Project.composeReloadDevToolsEnabled: Boolean get() {
    return composeReloadDevToolsEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.DevToolsTransparencyEnabled]
* Some platforms might not be able to render transparency correctly (e.g. some linux environments).
* This property will allow such platforms to disable/enable transparency
*/
@InternalHotReloadGradleApi val Project.composeReloadDevToolsTransparencyEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.devToolsTransparencyEnabled") 
        .orElse(providers.systemProperty("compose.reload.devToolsTransparencyEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.devToolsTransparencyEnabled")) 
        .orElse((Os.currentOrNull() != Os.Linux).toString()) 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.DevToolsTransparencyEnabled]
* Some platforms might not be able to render transparency correctly (e.g. some linux environments).
* This property will allow such platforms to disable/enable transparency
*/
@InternalHotReloadGradleApi val Project.composeReloadDevToolsTransparencyEnabled: Boolean get() {
    return composeReloadDevToolsTransparencyEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.JetBrainsRuntimeBinary]
* The path to the 'JetBrainsRuntime' which shall be used when launching the app. 
* Note: This is a build-only property!
*/
@InternalHotReloadGradleApi val Project.composeReloadJetBrainsRuntimeBinaryProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.jbr.binary") 
        .orElse(providers.systemProperty("compose.reload.jbr.binary")) 
        .orElse(providers.environmentVariable("compose.reload.jbr.binary")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.JetBrainsRuntimeBinary]
* The path to the 'JetBrainsRuntime' which shall be used when launching the app. 
* Note: This is a build-only property!
*/
@InternalHotReloadGradleApi val Project.composeReloadJetBrainsRuntimeBinary: Path? get() {
    return composeReloadJetBrainsRuntimeBinaryProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.AutoRuntimeDependenciesEnabled]
* Whether or not the hot-reload-runtime will be added as a compile dependency automatically when running a build.
*/
@InternalHotReloadGradleApi val Project.composeReloadAutoRuntimeDependenciesEnabledProvider: Provider<Boolean> get() {
    return providers.gradleProperty("compose.reload.autoRuntimeDependenciesEnabled") 
        .orElse(providers.systemProperty("compose.reload.autoRuntimeDependenciesEnabled")) 
        .orElse(providers.environmentVariable("compose.reload.autoRuntimeDependenciesEnabled")) 
        .orElse("true") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.AutoRuntimeDependenciesEnabled]
* Whether or not the hot-reload-runtime will be added as a compile dependency automatically when running a build.
*/
@InternalHotReloadGradleApi val Project.composeReloadAutoRuntimeDependenciesEnabled: Boolean get() {
    return composeReloadAutoRuntimeDependenciesEnabledProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IdeaComposeHotReload]
* Set by IntelliJ to signal the Gradle Plugin that IDE tooling is available.
* Setting this variable will relax the Gradle Plugin to not touch existing run tasks as we expect
* the IDE to provide a convenient way of launching in hot-reload mode.
*/
@InternalHotReloadGradleApi val Project.composeReloadIdeaComposeHotReloadProvider: Provider<Boolean> get() {
    return providers.gradleProperty("idea.compose.hot-reload") 
        .orElse(providers.systemProperty("idea.compose.hot-reload")) 
        .orElse(providers.environmentVariable("idea.compose.hot-reload")) 
        .orElse("false") 
        .map { raw -> raw.toBooleanStrict() } 
}

/**
* See [HotReloadProperty.IdeaComposeHotReload]
* Set by IntelliJ to signal the Gradle Plugin that IDE tooling is available.
* Setting this variable will relax the Gradle Plugin to not touch existing run tasks as we expect
* the IDE to provide a convenient way of launching in hot-reload mode.
*/
@InternalHotReloadGradleApi val Project.composeReloadIdeaComposeHotReload: Boolean get() {
    return composeReloadIdeaComposeHotReloadProvider 
        .get() 
}


  /**
* See [HotReloadProperty.IdeaComposeHotReloadSupportVersion]
* Set by IntelliJ during builds to convey its 'support level' for hot reload.
* Not Present, but 'idea.compose.hot-reload' being set: Supports running hot run tasks
* 
* 2: Support running 'hot run' tasks and 'hot reload' tasks
*/
@InternalHotReloadGradleApi val Project.composeReloadIdeaComposeHotReloadSupportVersionProvider: Provider<Int> get() {
    return providers.gradleProperty("idea.compose.hot-reload.version") 
        .orElse(providers.systemProperty("idea.compose.hot-reload.version")) 
        .orElse(providers.environmentVariable("idea.compose.hot-reload.version")) 
        .map { raw -> raw.toInt() } 
}

/**
* See [HotReloadProperty.IdeaComposeHotReloadSupportVersion]
* Set by IntelliJ during builds to convey its 'support level' for hot reload.
* Not Present, but 'idea.compose.hot-reload' being set: Supports running hot run tasks
* 
* 2: Support running 'hot run' tasks and 'hot reload' tasks
*/
@InternalHotReloadGradleApi val Project.composeReloadIdeaComposeHotReloadSupportVersion: Int? get() {
    return composeReloadIdeaComposeHotReloadSupportVersionProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StdinFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to the stdin file (can be pipe)
*/
@InternalHotReloadGradleApi val Project.composeReloadStdinFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stdinFile") 
        .orElse(providers.systemProperty("compose.reload.stdinFile")) 
        .orElse(providers.environmentVariable("compose.reload.stdinFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StdinFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to the stdin file (can be pipe)
*/
@InternalHotReloadGradleApi val Project.composeReloadStdinFile: Path? get() {
    return composeReloadStdinFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StdoutFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stdout is supposed to be written to.
*/
@InternalHotReloadGradleApi val Project.composeReloadStdoutFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stdoutFile") 
        .orElse(providers.systemProperty("compose.reload.stdoutFile")) 
        .orElse(providers.environmentVariable("compose.reload.stdoutFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StdoutFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stdout is supposed to be written to.
*/
@InternalHotReloadGradleApi val Project.composeReloadStdoutFile: Path? get() {
    return composeReloadStdoutFileProvider 
        .getOrNull() 
}


  /**
* See [HotReloadProperty.StderrFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stderr is supposed to be written to.
*/
@InternalHotReloadGradleApi val Project.composeReloadStderrFileProvider: Provider<Path> get() {
    return providers.gradleProperty("compose.reload.stderrFile") 
        .orElse(providers.systemProperty("compose.reload.stderrFile")) 
        .orElse(providers.environmentVariable("compose.reload.stderrFile")) 
        .map { raw -> Path(raw) } 
}

/**
* See [HotReloadProperty.StderrFile]
* Used by 'async'/'non-blocking' launches of the application.
* Will point to a file where the stderr is supposed to be written to.
*/
@InternalHotReloadGradleApi val Project.composeReloadStderrFile: Path? get() {
    return composeReloadStderrFileProvider 
        .getOrNull() 
}
