package org.jetbrains.compose.reload.core
import java.io.File
import java.nio.file.Path
import kotlin.io.path.Path
import org.jetbrains.compose.reload.core.Os
import org.jetbrains.compose.reload.InternalHotReloadApi

@InternalHotReloadApi
public object HotReloadEnvironment {
    /**
    * See [HotReloadProperty.OrchestrationPort]
    * Current 'Orchestration Server' port. This property is used by components that shall connect to the 
    * 'orchestration' (e.g. recompiler, devtools, ...)
    * 
    */
    public val orchestrationPort: Int? get() {
        val value = System.getProperty("compose.reload.orchestration.port")
           ?: System.getenv("compose.reload.orchestration.port")
            ?: return null
        return value.toInt()
    }
    
    
    /**
    * See [HotReloadProperty.PidFile]
    * Path to the current applications pidfile
    * 
    */
    public val pidFile: Path? get() {
        val value = System.getProperty("compose.reload.pidFile")
           ?: System.getenv("compose.reload.pidFile")
            ?: return null
        return Path(value)
    }
    
    
    /**
    * See [HotReloadProperty.ArgFile]
    * The java 'argfile' (see: https://docs.oracle.com/en/java/javase/21/docs/specs/man/java.html#java-command-line-argument-files)
    * containing all arguments of the current run
    * 
    */
    public val argFile: Path? get() {
        val value = System.getProperty("compose.reload.argfile")
           ?: System.getenv("compose.reload.argfile")
            ?: return null
        return Path(value)
    }
    
    
    /**
    * See [HotReloadProperty.MainClass]
    * Property available in the application and 'devtools' pointing to the mainClass used to launch the application
    * 
    */
    public val mainClass: String? get() {
        val value = System.getProperty("compose.reload.mainClass")
           ?: System.getenv("compose.reload.mainClass")
            ?: return null
        return value
    }
    
    
    /**
    * See [HotReloadProperty.IsHeadless]
    * Indicating whether or not the application is supposed to run in headless mode
    * 
    */
    public val isHeadless: Boolean get() {
        val value = System.getProperty("compose.reload.headless")
           ?: System.getenv("compose.reload.headless")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.HotClasspath]
    * The classpath known to be hot. This property is optional
    * 
    */
    public val hotClasspath: String? get() {
        val value = System.getProperty("compose.reload.hotApplicationClasspath")
           ?: System.getenv("compose.reload.hotApplicationClasspath")
            ?: return null
        return value
    }
    
    
    /**
    * See [HotReloadProperty.VirtualMethodResolveEnabled]
    * true: Enable dependency analysis for virtual calls:
    * e.g. Interfaces and their corresponding implementations will be tracked 
    * 
    */
    public val virtualMethodResolveEnabled: Boolean get() {
        val value = System.getProperty("compose.reload.virtualMethodResolveEnabled")
           ?: System.getenv("compose.reload.virtualMethodResolveEnabled")
            ?: "true"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.DirtyResolveDepthLimit]
    * If a given scope is marked as dirty, then Hot Reload will follow the dependency graph between fields/methods
    * and further marks scopes as dirty. This property limits how deep this graph can be traversed.
    * 
    */
    public val dirtyResolveDepthLimit: Int get() {
        val value = System.getProperty("compose.reload.dirtyResolveDepthLimit")
           ?: System.getenv("compose.reload.dirtyResolveDepthLimit")
            ?: "5"
        return value.toInt()
    }
    
    
    /**
    * See [HotReloadProperty.BuildSystem]
    * Indicating the application which build system is supposed to be used for recompiling.
    * See further build-system specific (Gradle, Amper, ...) properties.
    * 
    */
    public val buildSystem: org.jetbrains.compose.reload.core.BuildSystem? get() {
        val value = System.getProperty("compose.reload.buildSystem")
           ?: System.getenv("compose.reload.buildSystem")
            ?: return null
        return enumValueOf<org.jetbrains.compose.reload.core.BuildSystem>(value)
    }
    
    
    /**
    * See [HotReloadProperty.GradleJavaHome]
    * The 'java home' used to run Gradle. The recompiler will pick the same java to launch 
    * the recompiler in order to avoid cache misses or other issues.
    * 
    */
    public val gradleJavaHome: Path? get() {
        val value = System.getProperty("org.gradle.java.home")
           ?: System.getenv("org.gradle.java.home")
            ?: return null
        return Path(value)
    }
    
    
    /**
    * See [HotReloadProperty.GradleBuildRoot]
    * The root path to the current Gradle project
    * 
    */
    public val gradleBuildRoot: Path? get() {
        val value = System.getProperty("gradle.build.root")
           ?: System.getenv("gradle.build.root")
            ?: return null
        return Path(value)
    }
    
    
    /**
    * See [HotReloadProperty.GradleBuildProject]
    * The gradle 'path' to the 'project' which is currently executed and needs recompiling.
    * e.g. ':app:' or ':' or ':someModule:composeApp'
    * 
    */
    public val gradleBuildProject: String? get() {
        val value = System.getProperty("gradle.build.project")
           ?: System.getenv("gradle.build.project")
            ?: return null
        return value
    }
    
    
    /**
    * See [HotReloadProperty.GradleBuildTask]
    * The name of the task which is supposed to be recompiled for hot reload. 
    * This is typically the name of a 'ComposeReloadHotClasspathTask' task. 
    * 
    */
    public val gradleBuildTask: String? get() {
        val value = System.getProperty("gradle.build.task")
           ?: System.getenv("gradle.build.task")
            ?: return null
        return value
    }
    
    
    /**
    * See [HotReloadProperty.GradleBuildContinuous]
    * - true: Compose Hot Reload will start a recompiler Gradle Daemon, which will continuously rebuilt/reload the project
    * by watching all  inputs to the build
    * - false: The user is expected to rebuild/reload manually by launching a task (or using tooling)
    * Continuous mode is subject to change and might be removed in the future.
    * 
    */
    public val gradleBuildContinuous: Boolean get() {
        val value = System.getProperty("compose.reload.build.continuous")
           ?: System.getenv("compose.reload.build.continuous")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.GradleWarmupEnabled]
    * - true: Compose Hot Reload will launch a warmup recompile request when the application is started, to ensure that
    * the recompiler Gradle daemon is running and ready to handle requests
    * - false: No warmup request will be sent, first reload request may take longer time
    * 
    */
    public val gradleWarmupEnabled: Boolean get() {
        val value = System.getProperty("compose.reload.build.warmup")
           ?: System.getenv("compose.reload.build.warmup")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.AmperBuildRoot]
    */
    public val amperBuildRoot: String? get() {
        val value = System.getProperty("amper.build.root")
           ?: System.getenv("amper.build.root")
            ?: return null
        return value
    }
    
    
    /**
    * See [HotReloadProperty.AmperBuildTask]
    */
    public val amperBuildTask: String? get() {
        val value = System.getProperty("amper.build.task")
           ?: System.getenv("amper.build.task")
            ?: return null
        return value
    }
    
    
    /**
    * See [HotReloadProperty.DevToolsEnabled]
    * Flag to disable the 'devtools' application entirely
    * 
    */
    public val devToolsEnabled: Boolean get() {
        val value = System.getProperty("compose.reload.devToolsEnabled")
           ?: System.getenv("compose.reload.devToolsEnabled")
            ?: "true"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.DevToolsIsHeadless]
    * Run the dev tools in headless mode (No UI window shown)
    * 
    */
    public val devToolsIsHeadless: Boolean get() {
        val value = System.getProperty("compose.reload.devToolsHeadless")
           ?: System.getenv("compose.reload.devToolsHeadless")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.DevToolsClasspath]
    * The classpath notation of the devTools application. 
    * Hot Reload will start DevTools in a different process and therefore needs the classpath
    * 
    */
    public val devToolsClasspath: List<Path>? get() {
        val value = System.getProperty("compose.reload.devToolsClasspath")
           ?: System.getenv("compose.reload.devToolsClasspath")
            ?: return null
        return value.split(File.pathSeparator).map(::Path)
    }
    
    
    /**
    * See [HotReloadProperty.DevToolsTransparencyEnabled]
    * Some platforms might not be able to render transparency correctly (e.g. some linux environments).
    * This property will allow such platforms to disable/enable transparency. This property is subject to change
    * if the issues with transparency rendering are resolved..
    * 
    */
    public val devToolsTransparencyEnabled: Boolean get() {
        val value = System.getProperty("compose.reload.devToolsTransparencyEnabled")
           ?: System.getenv("compose.reload.devToolsTransparencyEnabled")
            ?: (Os.currentOrNull() != Os.Linux).toString()
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.DevToolsDetached]
    * If enabled, dev tools window will be detached from the main application
    * 
    */
    public val devToolsDetached: Boolean get() {
        val value = System.getProperty("compose.reload.devToolsDetached")
           ?: System.getenv("compose.reload.devToolsDetached")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.DevToolsAnimationsEnabled]
    * If disabled, dev tools will not use window animations when switching between expanded/minimised states. 
    * Note: window contents will still be animated. This property is present because of some platform-specific issues with
    * window management. Subject to change and might be removed in the future, if the current issues are resolved.
    * 
    */
    public val devToolsAnimationsEnabled: Boolean get() {
        val value = System.getProperty("compose.reload.devToolsAnimationsEnabled")
           ?: System.getenv("compose.reload.devToolsAnimationsEnabled")
            ?: "true"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.ReloadEffectsEnabled]
    * Enable reload effects that are shown in the UI when a reload is triggered.
    * 
    */
    public val reloadEffectsEnabled: Boolean get() {
        val value = System.getProperty("compose.reload.effectsEnabled")
           ?: System.getenv("compose.reload.effectsEnabled")
            ?: "true"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.IntelliJDebuggerDispatchPort]
    * Note: Expected as an environment variable, as this is expected to be transitively available
    * to all child processes.\n
    * Currently, launching applications with hot reload might be done through a couple of
    * intermediate processes. For example, launching a test will go through a chain like
    * ```
    * intellij --launches--> Gradle --launches--> JVM(Junit) --launches--> Gradle
    * --launches--> JVM (Application)
    * ```
    * When a run configuration is started in 'debug mode' intellij will set the system property
    * 'idea.debugger.dispatch.port'. This will indicate that a server is listening at this port, which can
    * be used to provision debugging servers.
    * This debug port will then be made available as an environment variable using this key.
    * Launching the final application will respect this port, if present and provision a debugging session.
    * This will allow a test to be deeply debuggable by just pressing 'Debug'
    * 
    */
    public val intelliJDebuggerDispatchPort: Int? get() {
        val value = System.getProperty("compose.reload.idea.debugger.dispatch.port")
           ?: System.getenv("compose.reload.idea.debugger.dispatch.port")
            ?: return null
        return value.toInt()
    }
    
    
    /**
    * See [HotReloadProperty.IdeaComposeHotReload]
    * Set by IntelliJ to signal the Gradle Plugin that IDE tooling is available.
    * Setting this variable will relax the Gradle Plugin to not touch existing run tasks as we expect
    * the IDE to provide a convenient way of launching in hot-reload mode.
    * 
    */
    public val ideaComposeHotReload: Boolean get() {
        val value = System.getProperty("idea.compose.hot-reload")
           ?: System.getenv("idea.compose.hot-reload")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.IdeaComposeHotReloadSupportVersion]
    * Set by IntelliJ during builds to convey its 'support level' for hot reload.
    * Not Present, but 'idea.compose.hot-reload' being set: Supports running hot run tasks
    * 
    * 2: Support running 'hot run' tasks and 'hot reload' tasks
    * 
    */
    public val ideaComposeHotReloadSupportVersion: Int? get() {
        val value = System.getProperty("idea.compose.hot-reload.version")
           ?: System.getenv("idea.compose.hot-reload.version")
            ?: return null
        return value.toInt()
    }
    
    
    /**
    * See [HotReloadProperty.StdinFile]
    * Used by 'async'/'non-blocking' launches of the application.
    * Will point to the stdin file (can be pipe)
    * 'async'/'non-blocking' mode is subject to change and might be removed in the future.
    * 
    */
    public val stdinFile: Path? get() {
        val value = System.getProperty("compose.reload.stdinFile")
           ?: System.getenv("compose.reload.stdinFile")
            ?: return null
        return Path(value)
    }
    
    
    /**
    * See [HotReloadProperty.StdoutFile]
    * Used by 'async'/'non-blocking' launches of the application.
    * Will point to a file where the stdout is supposed to be written to.
    * 'async'/'non-blocking' mode is subject to change and might be removed in the future.
    * 
    */
    public val stdoutFile: Path? get() {
        val value = System.getProperty("compose.reload.stdoutFile")
           ?: System.getenv("compose.reload.stdoutFile")
            ?: return null
        return Path(value)
    }
    
    
    /**
    * See [HotReloadProperty.StderrFile]
    * Used by 'async'/'non-blocking' launches of the application.
    * Will point to a file where the stderr is supposed to be written to.
    * 'async'/'non-blocking' mode is subject to change and might be removed in the future.
    * 
    */
    public val stderrFile: Path? get() {
        val value = System.getProperty("compose.reload.stderrFile")
           ?: System.getenv("compose.reload.stderrFile")
            ?: return null
        return Path(value)
    }
    
    
    /**
    * See [HotReloadProperty.ParentPid]
    * The pid of the parent process. This property is used by components that shall connect to the 
    * 'orchestration' (e.g. recompiler, devtools, ...).
    * If the parent dies, it is expected that children exit as well.
    * 
    */
    public val parentPid: Long? get() {
        val value = System.getProperty("compose.reload.parentPid")
           ?: System.getenv("compose.reload.parentPid")
            ?: return null
        return value.toLong()
    }
    
    
    /**
    * See [HotReloadProperty.LaunchMode]
    * Tells the application 'how' it was launched
    * 
    */
    public val launchMode: org.jetbrains.compose.reload.core.LaunchMode? get() {
        val value = System.getProperty("compose.reload.launchMode")
           ?: System.getenv("compose.reload.launchMode")
            ?: return null
        return enumValueOf<org.jetbrains.compose.reload.core.LaunchMode>(value)
    }
    
    
    /**
    * See [HotReloadProperty.LogLevel]
    * Minimum logging level
    * 
    */
    public val logLevel: org.jetbrains.compose.reload.core.Logger.Level get() {
        val value = System.getProperty("compose.reload.logLevel")
           ?: System.getenv("compose.reload.logLevel")
            ?: "Info"
        return enumValueOf<org.jetbrains.compose.reload.core.Logger.Level>(value)
    }
    
    
    /**
    * See [HotReloadProperty.LogStdout]
    * Enable output of all logs into the standard output
    * 
    */
    public val logStdout: Boolean get() {
        val value = System.getProperty("compose.reload.logStdout")
           ?: System.getenv("compose.reload.logStdout")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
    /**
    * See [HotReloadProperty.IsHotReloadActive]
    * Will be set to 'true' if the application is launched with Hot Reload and therefore can be used
    * to detect if hot reload is 'active'
    * 
    */
    public val isHotReloadActive: Boolean get() {
        val value = System.getProperty("compose.reload.isActive")
           ?: System.getenv("compose.reload.isActive")
            ?: "false"
        return value.toBooleanStrict()
    }
    
    
}